/******************************************
DUCK PUNT
Copyright (C) 2005 Geoffrey M. Draper

This file is part of Duck Punt.
Duck Punt is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************/

/*****************************************
VERSION HISTORY

Version 1.0 (March 2005) - Initial release
Version 1.1 (January 2006) - F2 to toggle sound
Version 1.2 (December 2007) - Bugfix (automatic key repeat)
******************************************/

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.lang.String;

public class DuckPunt extends JPanel implements MouseListener, KeyListener
{
	Timer timer;
	Player player[];
	Duck duck;
	int clock;
	private int step_val;
	int yardage_when_kicked = 0;
	Color bluePlayerColor = new Color(62,101,178);
	Color redPlayerColor = new Color(198,5,60);
	static final int delay = 100;
	static final int GAME_LENGTH = 60;
	public int current_player;
	public int window_width;
	public int window_height;
	public int game_mode;
	public MathProblem math;
	private long timestampOfLastKeyPress;
	private static final long KEY_REPEAT_THRESHOLD = 10;
	public final int grass_depth=75;
	public static final int pixelsToYard = 100;
	public static final int MATH_MODE = 0;
	public static final int SIMPLE_MODE = 1;
	
	public DuckPunt()
	{

		//display splash screen here
		SplashScreen splash = new SplashScreen();

		math = new MathProblem();

		timer = new Timer(delay, new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				int diff = duck.getX() - player[current_player].getX();
				switch (player[current_player].getStatus())
				{
					case Player.RUNNING:
					duck.moveRelativeToPlayer();
					player[current_player].advanceFrame(duck.willBeIntercepted());
					
					if (game_mode == MATH_MODE)
					{
						//make player fall flat on back if answer is too low
						if (math.getRelativeAnswer() < 0)
						{
							if (diff < 300)
							{
								player[current_player].kick();
								makePlayerFallOnBack();
								break;
							}
						}
						//kick the duck if the answer is right
						if (math.getRelativeAnswer() == 0)
						{
							if (diff < 100)
							{
								player[current_player].kick();
								setupPunt(diff,200);
								//in math mode, these numbers are cosmetic only
							}						
						}
					}
					//trip over the duck if either the answer is too high,
					//OR if the user just forgot to kick
					if (((diff < -50)) &&
						(game_mode == SIMPLE_MODE || (game_mode == MATH_MODE && math.getRelativeAnswer() > 0)))
					{
						SoundEffects.play(SoundEffects.FALL_SOUND);
						player[current_player].spring_forward();
					}
					break;
					
					case Player.RUNNING_IN_PLACE:
					player[current_player].advanceFrame(duck.willBeIntercepted());
					break;

					//in this 'case', the intercepter has caught the duck, and they
					//are rendered together as a single image (duck sprite disappears)
					case Player.AFTER_INTERCEPTION:
					player[(current_player + 1) % 2].advanceFrame_carryingDuck();
					//if intercepting player moves off-screen, then the turn is over
					//and control reverts to intercepting player
					if (player[(current_player + 1) % 2].getX() < -100)
					{
						duck.setYardage(100-duck.getYardage());
						switchPlayers();
						returnToHomePosition();
						decrementClockAndRepaint();
					}
					break;
					
					//in this 'case', the both the spinning duck and the intercepter
					//are on-screen, but the intercepter has not yet caught the duck
					case Player.BEFORE_INTERCEPTION:
					player[(current_player + 1) % 2].advanceFrame(true);
					if (duck.isReadyToIntercept())
					{
						SoundEffects.play(SoundEffects.WHISTLE_SOUND);
						player[current_player].setStatus(Player.AFTER_INTERCEPTION);
						break;
					}

					//in this 'case', the duck is spinning through the screen,
					//but the intercepter has not yet appeared on-screen
					case Player.DUCK_FLYING:
					//the punter moves backwards, creating the illustion that the
					//duck is moving forwards
					player[current_player].fadeAway(step_val);
					
					//can we start the intercepter running yet?
					if (duck.isAlmostReadyToIntercept())
					{
						SoundEffects.play(SoundEffects.CHEER_SOUND);
						player[current_player].setStatus(Player.BEFORE_INTERCEPTION);
					}
					
					if (!duck.animate())
					{
						player[current_player].setStatus(Player.DUCK_LANDED);
						timer.stop();
					}
					break;
				}
				repaint();
			}
		});
        
		timer.setInitialDelay(0);
		timer.setCoalesce(true);
		
		addMouseListener(this);
		initializeVariables(splash);
	}
	
	private void initializeVariables(SplashScreen splash)
	{
		//By default, we size the window so that it will fit
		//on 800x600 resolution displays.  Users are welcome
		//to resize the window for their individual displays.
		window_width = 799;
		window_height =	500;
		Dimension dim = new Dimension(window_width, window_height);
		setPreferredSize(dim);

		timestampOfLastKeyPress = 0;
		step_val = 25;
		player = new Player[2];
		player[0] = new Player(0, "ROJO");
		player[1] = new Player(1, "AZUL");
		duck = new Duck(step_val);
		//SoundEffects = new SoundEffects();
		SoundEffects.init();
		
		//make sure the splash screen stays up for at least 3 seconds
		splash.closeAfterAppropriateTimeout(3000);
		
		setDefaults(0);	//make player 0 go first
		
		//move the duck farther away from player for the first round
		//to give player time to understand the delicate timing of the
		//"Click to Run, Click to Punt" routine...
		duck.padForBeginners(500);
	}

	private void setDefaults(int next_player)
	{
		game_mode = SIMPLE_MODE; //default
		
		timer.setDelay(delay);
		duck.setYardage(90);
		clock = GAME_LENGTH;
		current_player = next_player;

		//show pop-up window to get game mode		
		Object[] options = {"Math Mode", "Regular Mode"};
		int n = JOptionPane.showOptionDialog(this, "You can choose either \"Math Mode\" or \"Regular Mode\"",
			"Welcome to DUCK PUNT",
			JOptionPane.YES_NO_OPTION,
			JOptionPane.PLAIN_MESSAGE,
			null,
			options,
			options[1]);
			
		if (n == JOptionPane.YES_OPTION) {
			game_mode = MATH_MODE;
			timer.start();
		}
		
		player[0].setDefaults(game_mode, window_width);
		player[1].setDefaults(game_mode, window_width);
		duck.returnToHomePosition(current_player, window_width);
		player[current_player].playSong();
	}

    public void paintComponent(Graphics g)
	{
		window_width = getWidth();
		window_height = getHeight();
		String textToRender;

		//draw grass and sky
		g.setColor(new Color(185,220,247));
		g.fillRect(0,0, window_width,window_height-grass_depth);
		g.setColor(new Color(10,162,114));
		g.fillRect(0,window_height-grass_depth, window_width,window_height);

		//display grid yardage
		displayGridYardage(g, window_height-grass_depth);

		//display time clock
		g.setFont(new Font("SansSerif", Font.BOLD, 20));
		g.setColor(Color.black);
		textToRender = "TIME: " + String.valueOf(clock);
		g.drawString(textToRender, (window_width - g.getFontMetrics().stringWidth(textToRender))/2,20);

		//display yards to goal
		g.setFont(new Font("SansSerif", Font.BOLD, 15));
		g.setColor(Color.black);
		textToRender = "YARDS TO GOAL: " + String.valueOf(duck.getYardage());
		g.drawString(textToRender, (window_width - g.getFontMetrics().stringWidth(textToRender))/2,40);

		//only draw player & duck if there's time left in the game
		if (clock > 0)
		{
			//draw player
			player[current_player].drawMe(this, g);
	
			//draw duck as long as other player isn't carrying it
			if (player[current_player].getStatus() != Player.AFTER_INTERCEPTION )
			{
				duck.drawMe(this, g);
			}
	
			//draw the opposing player only if an interception is occurring
			if (player[current_player].getStatus() == Player.BEFORE_INTERCEPTION ||
				player[current_player].getStatus() == Player.AFTER_INTERCEPTION )
			{
				player[(current_player + 1) % 2].drawMe(this, g);
			}	
			displayTextMessage(g);
		}
		
		//display score information
		g.setFont(new Font("SansSerif", Font.BOLD, 30));
		g.setColor(redPlayerColor);
		textToRender = player[0].getName() + ": " + String.valueOf(player[0].getScore());
		g.drawString(textToRender, 10,30);
		g.setColor(bluePlayerColor);
		textToRender = player[1].getName() + ": " + String.valueOf(player[1].getScore());
		g.drawString(textToRender, window_width-10-g.getFontMetrics().stringWidth(textToRender),30);

	}
	
	private void displayTextMessage(Graphics g)
	{
		String textToRender1="", textToRender2="";
		String fontStyle = "SansSerif";
		int textX, textY;
		Font font1, font2, tmpfont;
		
		if (game_mode == MATH_MODE) {
			fontStyle = "Monospaced";
		}
		font1 = new Font(fontStyle, Font.BOLD, 30);
		font2 = new Font(fontStyle, Font.PLAIN, 15);
		
		if (current_player == 0) g.setColor(redPlayerColor);
		else g.setColor(bluePlayerColor);

		switch(player[current_player].getStatus())
		{
			case Player.STANDING:
			if (game_mode == SIMPLE_MODE)
			{
				textToRender1 = "CLICK TO RUN";
				textToRender2 = "CLICK TO PUNT";
			}
			break;

			case Player.RUNNING:
			case Player.RUNNING_IN_PLACE:
			if (game_mode == MATH_MODE)
			{
				textToRender1 = math.getProblemText();
			} else {
				textToRender1 = "CLICK TO RUN";
				textToRender2 = "CLICK TO PUNT";
				//switch fonts
				tmpfont = font1;
				font1 = font2;
				font2 = tmpfont;
			}
			break;

			case Player.DUCK_LANDED:
			if (duck.getYardage() <= 0)
			{
				textToRender1 = "TOUCHDOWN!";
				player[current_player].incrementScore(6);
				player[current_player].setStatus(Player.TOUCHDOWN);
				SoundEffects.play(SoundEffects.CHEER_SOUND);
			} else
			{
				textToRender1 = String.valueOf(duck.getPuntLength()) + " YARD PUNT!";
			}
			textToRender2 = "CLICK TO CONTINUE";
			break;

			case Player.SPRING_FORWARD:
			case Player.FALL_BACK:
			if (game_mode == MATH_MODE)
			{
				textToRender1 = "Sorry, the correct answer is:";
			} else {
				textToRender1 = "WHOOPS!";
			}
			timer.stop();
			textToRender2 = "CLICK TO CONTINUE";
			break;

			case Player.BEFORE_INTERCEPTION:
			case Player.AFTER_INTERCEPTION:
			textToRender1 = "INTERCEPTION!";
			break;
			
			default:
			textToRender1 = "";
			textToRender2 = "";
			break;

		}

		g.setFont(font1);
		textX = (window_width - g.getFontMetrics().stringWidth(textToRender1))/2;
		textY = (window_height / 3);
		g.drawString(textToRender1, textX, textY);

		//render additional text if in math mode
		if (game_mode == MATH_MODE)
		{
			switch (player[current_player].getStatus())
			{
				case Player.RUNNING_IN_PLACE:
				case Player.RUNNING:
				//set pen position to end of string
				textX = (window_width + g.getFontMetrics().stringWidth(textToRender1))/2;
				//draw question-marks, or the user's input
				g.setColor(Color.white);
				//re-use the textToRender variable since we're otherwise done with it
				textToRender1 = math.getUserResponseText();
				g.drawString(textToRender1, textX, textY);
				break;

				case Player.SPRING_FORWARD:
				case Player.FALL_BACK:
				textToRender1 = math.getCompleteSolution();
				textX = (window_width - g.getFontMetrics().stringWidth(textToRender1))/2;
				textY += (g.getFontMetrics().getHeight())/1.5;
				g.drawString(textToRender1, textX, textY);
				break;

			}
		}
		
		//draw textToRender2 if needed
		textY += (int)((float)(g.getFontMetrics().getHeight()) / ((font1.getSize() == 30) ? 2.0 : 0.66));
		g.setFont(font2);
		textX = (window_width - g.getFontMetrics().stringWidth(textToRender2))/2;
		g.drawString(textToRender2, textX, textY);
	}

	private void displayGridYardage(Graphics g, int ground_level)
	{
		int absoluteYardage;
		int gridyard;
		int x;
		String textToRender;
		g.setColor(Color.white);
		g.setFont(new Font("SansSerif", Font.BOLD, 30));

		//draw field the same regardless of who the current player is
		if (player[current_player].getStatus() == Player.DUCK_FLYING ||
			player[current_player].getStatus() == Player.AFTER_INTERCEPTION ||
			player[current_player].getStatus() == Player.BEFORE_INTERCEPTION ||
			player[current_player].getStatus() == Player.DUCK_LANDED) {
			if (current_player == 0) absoluteYardage=100-yardage_when_kicked;
			else absoluteYardage=yardage_when_kicked;
		} else {
			if (current_player == 0) absoluteYardage=100-duck.getYardage();
			else absoluteYardage=duck.getYardage();
		}

		//the zero-yard line is placed relative to the duck's position
		//we start rendering from the zero-yard line, even if it's off the screen
		if (current_player == 0) {
			x = duck.getX() - (absoluteYardage * pixelsToYard);
		}
		else
		{
			x = window_width - duck.getX() - (absoluteYardage * pixelsToYard);
		}
		for (int i=0; i<=100; i++)
		{
			//numbers are displayed from 0 to 50, and from 50 to 0
			if (i<50) gridyard=i;
			else gridyard=100-i;

			//every 5 yards, draw a marker
			//every 10 yards, draw a number
			if (gridyard % 5 == 0) g.fillRect(x-4, ground_level, 8,12);
			if (gridyard % 10 == 0)
			{
				textToRender = String.valueOf(gridyard);
				g.drawString(textToRender, x-g.getFontMetrics().stringWidth(textToRender)/2, ground_level+40);
			}
			x += pixelsToYard;
		}
	}
	
	//draws the player on his back, after kicking too early
	public void makePlayerFallOnBack()
	{
		SoundEffects.play(SoundEffects.SLIP_SOUND);
		update(getGraphics());
		pause(1500);
		SoundEffects.play(SoundEffects.FALL_SOUND);
		timer.stop();
		player[current_player].fall_back();				
		repaint();
	}

	//respond to mouse-click events (or key events that we've forwarded here)
	public void mousePressed(MouseEvent e)
	{
		//don't respond to user events if the time is up
		if (clock > 0)
		{
			int diff;
			final int KICKING_RANGE = 200;
			switch(player[current_player].getStatus())
			{
				case Player.STANDING:
				player[current_player].setStatus(Player.RUNNING);
				timer.setDelay(delay);
				timer.start();
				break;
	
				case Player.RUNNING:
				if (game_mode == SIMPLE_MODE)
				{
					player[current_player].kick();
					diff = duck.getX() - player[current_player].getX() + 50;//50=fudge_factor
		
					if (diff > KICKING_RANGE)
					{
						//fall flat on back
						makePlayerFallOnBack();
						break;
					}
		
					setupPunt(diff, KICKING_RANGE);
				}
				break;
	
				case Player.TOUCHDOWN:
				switchPlayers();
				duck.setYardage(90);
				//don't "break;" continue on into the next case
	
				case Player.DUCK_LANDED:
				returnToHomePosition();
				decrementClockAndRepaint();
				break;
	
				case Player.SPRING_FORWARD:
				case Player.FALL_BACK:
				duck.setYardage(100-duck.getYardage());
				switchPlayers();
				returnToHomePosition();
				decrementClockAndRepaint();
				break;
	
				default:
				break;
			}
		}
	}
	
	public void switchPlayers()
	{
		player[current_player].stopSong();
		current_player = ((current_player + 1) % 2);
		player[current_player].playSong();		
	}
	
	public void decrementClockAndRepaint()
	{
		clock--;
		repaint();
		checkForEndGame();
	}
	
	//This function returns both duck and player to their default
	//screen positions, and creates a new math problem (in math mode)
	public void returnToHomePosition()
	{
		player[current_player].returnToHomePosition(game_mode, window_width);
		duck.returnToHomePosition(current_player, window_width);
		
		if (game_mode == MATH_MODE) 
		{
			math.makeNewProblem();
			timer.setDelay(delay);
			timer.start();
		}
		else
		{
			timer.stop();
		}
	}
	
	//This function sets all the needed variables prior to a punt
	public void setupPunt(int diff, int range)
	{
		SoundEffects.play(SoundEffects.KICK_SOUND);
		
		//force an immediate screen refresh
		update(getGraphics());

		player[current_player].setStatus(Player.DUCK_FLYING);
		if (game_mode == MATH_MODE)
		{
			duck.setupPunt(diff, range, math.getAnswer(), window_width);
		} else {
			duck.setupPunt(diff, range, -1, window_width);
			if (duck.willBeIntercepted() == true)
			{
				player[(current_player + 1) % 2].prepareToIntercept(window_width);
			}
		}
		yardage_when_kicked = duck.getYardage();
		timer.setDelay(delay/2);
	}
	
	public void checkForEndGame()
	{
		if (clock <= 0)
		{
			int next_player = 1;
			player[current_player].stopSong();
			SoundEffects.play(SoundEffects.CHEER_SOUND);
			timer.stop();
			String textToRender;
			textToRender = "IT'S A DRAW!";
			if (player[0].getScore() > player[1].getScore()) {
				textToRender = player[0].getName() + " WINS!!";
			}
			if (player[1].getScore() > player[0].getScore()) {
				textToRender = player[1].getName() + " WINS!!";
				next_player = 0;
			}
			textToRender += "  PLAY AGAIN?";

			//pop up dialog box
			int n = JOptionPane.showOptionDialog(this, textToRender,
				"GAME OVER",
				JOptionPane.YES_NO_OPTION,
				JOptionPane.PLAIN_MESSAGE,
				null,
				null,
				null);
			if (n == JOptionPane.YES_OPTION) {
				setDefaults(next_player);
				repaint();
			} else {
				System.exit(0);
			}
		}

	}

	public void mouseEntered(MouseEvent e) {}

	public void mouseExited(MouseEvent e) {}

	public void mouseClicked(MouseEvent e) {}

	public void mouseReleased(MouseEvent e) {}
	
	public void keyPressed(KeyEvent e)
	{
		long timestamp = System.currentTimeMillis();
		long timeSinceLastKeyPress = timestamp - timestampOfLastKeyPress;
		timestampOfLastKeyPress = timestamp;
		//System.out.println("milleeconds since last key press: " + timeSinceLastKeyPress);
		if (timeSinceLastKeyPress < 50) return;
		
		//quit game if ESCAPE key is pressed
		if (e.getKeyCode() == KeyEvent.VK_ESCAPE)
		{
			System.exit(0);
		}
		
		//toggle sound if F2 key is pressed
		if (e.getKeyCode() == KeyEvent.VK_F2)
		{
			SoundEffects.toggleSound();
			if (SoundEffects.enabled()) {
				player[current_player].playSong();
			} else {
				player[current_player].stopSong();
			}
			return;
		}
	
		if (clock > 0)
		{
			switch(game_mode)
			{
				//in simple mode, treat all keystrokes as if they were mouseclicks
				case SIMPLE_MODE:
				mousePressed(new MouseEvent(e.getComponent(),e.getID(),e.getWhen(),e.getModifiers(),0,0,0,false));
				break;
				
				case MATH_MODE:
				switch (player[current_player].getStatus())
				{
					case Player.RUNNING_IN_PLACE:
					boolean done;
					done = math.handleKeyStrokes(e);
					if (done)
					{
						//evaluate response
						player[current_player].setStatus(Player.RUNNING);
					}
					break;
					
					case Player.RUNNING:
					//do nothing
					
					//let the mousePressed function handle all other key events
					default:
					mousePressed(new MouseEvent(e.getComponent(),e.getID(),e.getWhen(),e.getModifiers(),0,0,0,false));
					break;			
				}
				break;
				
				default:
				System.out.println("keyPressed event received; unknown game_mode.");
				break;
			
			}
		}
	}

	public void keyReleased(KeyEvent e) {}
    
	public void keyTyped(KeyEvent e) {}

	public static void pause(long m)
	{
		try {
			Thread.sleep(m);
		} catch (InterruptedException e) {System.out.println("pause(): exception caught");}
	}

	public static void main(String[] args) {
	
		//have look-n-feel mimic that of the operating system, if possible
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) { }

		JFrame frame = new JFrame("DUCK PUNT");

		//if using Java 1.2
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				System.exit(0);
			}
		});
		//if using Java 1.4:
		//frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		ImageIcon icon = new ImageIcon("data/images/duck_small_icon.gif");
		frame.setIconImage(icon.getImage());

		DuckPunt app = new DuckPunt();

		frame.getContentPane().add(app, BorderLayout.CENTER);
		frame.addKeyListener(app);
		frame.pack();
		frame.setVisible(true);
	}
}
