#ifndef WIN32
/******************************************

GetProcByName - copyright 2003 John Dawson

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

******************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <utmp.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>
#include <pwd.h>
#include <sys/ioctl.h>
#include <string.h>

#include "GetProcByName.h"

extern int errno;

#ifndef SPACE
#define SPACE	(char)0x20
#endif

/*****************************************************************************
** GetProcByName - function searches the process table looking for a process
**   whose command line contains the string contained int the ProcName
**   argument.
**
** Arguments:
**   char *ProcName [in] - contains the string to match.
**   pid_t *PID [out] - If non-NULL, will contain the PID of the first process
**      which matches the search criteria.
**   char *ErrorText [out] - If non-NULL and ErrorTextSize is greater than 1,
**      argument will contain any error messages generated by this function.
**   int ErrorTextSize [in] - length in bytes of the ErrorText argument.
**
** Returns:
**   0 on success
**   -1 on error
**   1 if no match is found
**
** Notes:
**   - strstr() is used to search for the ProcName string within the command
**     line text of each process. Therefore a non-unique string can cause a
**     false match. For example, if the "at" daemon (atd) is the process you
**     are searching for, the rpc.statd process can give a false positive since
**     it also contains the string "atd".
**   - We filter out ourselves and our immediate parent from the search, so
**     if our process is named "thatdubiousprogram", it will not cause a false
**     positive even though it also contains the text "atd".
**   - The ProcName string is checked against the full command line of each
**     process, including any arguments passed.
**     This means that if you are searching for the text "deamon", you can
**     get a match on the process "/usr/bin/gdm -nodaemon".
*****************************************************************************/
int
GetProcByName(char *ProcName, pid_t *PID, char *ErrorText, int ErrorTextSize)
{
	DIR *dptr;
	FILE *procfptr;
	struct dirent *dirbuf;
	int bytes_r=0, count=0;
	int cmdfd=0;
	char cmdfname[25], cmdbuf[257];
	char procfname[25];
	char cwd[257];
	char *tptr;
	struct stat statbuf;
	struct PROC procptr;
	int found=0;
	int DoErrorText=1;
	pid_t mypid, myppid;

	//get current working directory
	if( getcwd(cwd, sizeof(cwd)) == (char *)NULL ) {
		return -1;
	}

	if( ErrorText == (char *)NULL || ErrorTextSize < 2 )
		DoErrorText=0;

	if( ProcName == (char *)NULL )
	{	if( DoErrorText )
			snprintf(ErrorText,ErrorTextSize,"Passed NULL pointer for ProcName");
		return -1;
	}

	if( (dptr=opendir("/proc")) == (DIR *)NULL )
	{	if( DoErrorText )
			snprintf(ErrorText,ErrorTextSize,"opendir error %d on /proc",errno);
		return -1;
	}

	mypid=getpid();
	myppid=getppid();
	chdir("/proc");

	/*
	** Main loop to read in specific process data from /proc.
	*/
	while( (dirbuf=readdir(dptr)) != (struct dirent *)NULL )
	{
		if( stat(dirbuf->d_name,&statbuf) != 0 )
		{	/* No longer a process? */
			continue;
		}
		if( ! S_ISDIR(statbuf.st_mode) )
			continue;

		if( atoi(dirbuf->d_name) <= 0 )
		{	/* Not a PID (or is process 0) */
			continue;
		}

		/*
		** First read the full proc structure. This gives us a chance
		** to filter out ourselves and anything else we want.
		** Additionally, we will need the cmdline information from this
		** structure for some processes because the cmdline "file" is
		** not always populated for all processes.
		*/
		snprintf(procfname,sizeof(procfname),"/proc/%s/stat",dirbuf->d_name);
		if( (procfptr=(FILE *)fopen(procfname,"r")) == (FILE *)NULL )
		{	/* Don't know why this would happen in /proc.
			** It's not a likely occurrence. Choosing to ignore it here.
			*/
			continue;
		}

		if( fscanf(procfptr,
			"%ld %s %c %ld %ld %d %d %d %ld %lu %lu %lu %lu %ld %ld %ld %ld %d %d %lu %ld %ld %lu %lu %lu %lu %lu %lu %lu %lu %LX %LX %LX %LX %lu %lu %lu",
			&procptr.sproc.pid,
			procptr.sproc.cmd,
			&procptr.sproc.state,
			&procptr.sproc.ppid,
			&procptr.sproc.pgrp,
			&procptr.sproc.session,
			&procptr.sproc.ttydev,
			&procptr.sproc.tpgid,
			&procptr.sproc.flags,
			&procptr.sproc.minflt,
			&procptr.sproc.cminflt,
			&procptr.sproc.majflt,
			&procptr.sproc.cmajflt,
			&procptr.sproc.utime,
			&procptr.sproc.stime,
			&procptr.sproc.cutime,
			&procptr.sproc.cstime,
			&procptr.sproc.counter,
			&procptr.sproc.priority,
			&procptr.sproc.timeout,
			&procptr.sproc.itrealvalue,
			&procptr.sproc.starttime,
			&procptr.sproc.vsize,
			&procptr.sproc.rss,
			&procptr.sproc.rlim,
			&procptr.sproc.startcode,
			&procptr.sproc.endcode,
			&procptr.sproc.startstack,
			&procptr.sproc.kstkesp,
			&procptr.sproc.kstkeip,
			&procptr.sproc.signal,
			&procptr.sproc.blocked,
			&procptr.sproc.sigignore,
			&procptr.sproc.sigcatch,
			&procptr.sproc.wchan,
			&procptr.sproc.filler1,
			&procptr.sproc.filler2) <= 0 )
		{	/* Choosing to ignore. May want to handle differently. */
			continue;
		}

		fclose(procfptr);

		/* Filter out ourselves and our parent. */
		/* Can do any additional filtering here */
		if( procptr.sproc.pid == mypid || procptr.sproc.pid == myppid )
			continue;

		/*
		** Read full command line from /proc/<pid>/cmdline.
		*/
		memset(cmdbuf,0,sizeof(cmdbuf));
		snprintf(cmdfname,sizeof(cmdfname),"/proc/%s/cmdline",dirbuf->d_name);
		if( (cmdfd=open(cmdfname,O_RDONLY)) >= 0 )
		{	if( (bytes_r=read(cmdfd,cmdbuf,sizeof(cmdbuf)-1)) > 0 )
			{	close(cmdfd);
				cmdbuf[bytes_r]=(char)NULL;
				/* Change embedded NULL's to spaces. */
				while( (tptr=(char *)index(cmdbuf,(int)NULL)) != (char *)NULL )
				{	if( tptr > (cmdbuf+((char) bytes_r)) )
						break;
					*tptr=SPACE;
				}
				/* Eliminate trailing spaces. */
				count=bytes_r-1;
				while( cmdbuf[count] == SPACE )
				{	cmdbuf[count]=(char)NULL;
					count--;
					if( count <= 0 )
					{	memset(cmdbuf,0,sizeof(cmdbuf));
						break;
					}
				}	/* End while() loop */
			}	/* End if() for reading cmdline */
		}	/* End if() for opening cmdline for process. */


		/*
		** It's possible that the cmdline entry under /proc contained no
		** data. In that event, we'll get the info from the proc structure.
		** Note that the info in the proc structure is only the command name,
		** not the full command line.
		*/
		if( cmdbuf[0] == (char)NULL )
			strncpy(cmdbuf,procptr.sproc.cmd,sizeof(cmdbuf)-1);


		/*
		** Now see if we have a match.
		*/
		if( (char *)strstr(cmdbuf,ProcName) == (char *)NULL )
		{	/* These are not the droids you are looking for */
			continue;
		}
		else
		{	found=1;
			if( PID != (pid_t *)NULL )
				*PID=(pid_t)procptr.sproc.pid;
			break;
		}
	}	/* End while() loop for reading directory entries */

	closedir(dptr);

	//go back to previous working directory (before /proc)
	chdir(cwd);
	
	if( found )
		return 0;
	return 1;
}	/* End function fill_native_proc_list() */
#endif //WIN32