package projman;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.Timer;
import javax.swing.event.MouseInputAdapter;

public abstract class AbstractWhiteboard extends JPanel {
	protected List<WhiteboardTeam> teams;
	protected Dimension windowSize;
	protected ChartStyle chartStyle;
	protected Rectangle lassoRect;
	protected WhiteboardMouseHandler mouse;
	private Timer flashTimer;
	private int flashCounter;
	private boolean flashMode;
	private boolean testing;
	//private JScrollPane scroller;

	public enum ChartStyle {
		HORIZONTAL,
		VERTICAL
	}

	public AbstractWhiteboard() {
		chartStyle = Config.DEFAULT_CHARTSTYLE;
		lassoRect = new Rectangle();
		resize();
		flashMode = false;
		testing = false;
		//setPreferredSize(new Dimension(0,0));
		//scroller = new JScrollPane(this);
		//scroller.setPreferredSize(new Dimension(1000,1000));
		//scroller.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);
	}

	protected abstract class WhiteboardMouseHandler extends MouseInputAdapter implements Serializable {    
		protected List<WhiteboardTeam> draggedTeams = new ArrayList<WhiteboardTeam>();
		protected Point prevMousePos, mousePos;
		protected Point lassoPos = null;
	}

	protected void resize() {
		windowSize = getSize();
	}

	public void setChartStyle(ChartStyle style) {
		if (chartStyle == style) return;
		chartStyle = style;
		repaint();
	}

	public ChartStyle getChartStyle() {
		return chartStyle;
	}

	@Override
	public void paintComponent(Graphics g1) {
		Graphics2D g = (Graphics2D)g1;
		if (!(windowSize.equals(getSize()))) {
			resize();
		}
		
		//This is NOT part of the "public interface".
		//The user should never be able to turn on test mode.
		if (testing) {
			arrangeTeamsNicely();
			testing = false;
		}

		//turn on antialiasing, if desired.
		if (Config.antialias) {
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		} else {
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
		}

		//set the line thickness
		g.setStroke(Config.regularLine);

		g.setColor(Config.backgroundColor);
		g.fillRect(0,0,windowSize.width, windowSize.height);
		for (WhiteboardTeam t : teams) {
			t.draw(g);
			//Main.say("Team " + t + " rendered at " + t.bounds);
		}

		//render any subclass-specific stuff
		drawAdditional(g);

		if (mouse.lassoPos != null && mouse.draggedTeams.isEmpty()) {
			g.setColor(Config.wb_lassoColor);
			if (mouse.mousePos.x < mouse.lassoPos.x) lassoRect.x = mouse.mousePos.x;
			if (mouse.mousePos.y < mouse.lassoPos.y) lassoRect.y = mouse.mousePos.y;
			int width = Math.abs(mouse.lassoPos.x - mouse.mousePos.x);
			int height = Math.abs(mouse.lassoPos.y - mouse.mousePos.y);
			lassoRect.setSize(width, height);
			g.drawRect(lassoRect.x, lassoRect.y, width, height);
		}
		if (!mouse.draggedTeams.isEmpty()) {
			g.setColor(Config.wb_selectedTeamBorderColor);
			g.drawRect(lassoRect.x, lassoRect.y, lassoRect.width, lassoRect.height);
		}
	}

	protected abstract void drawAdditional(Graphics2D g);

	public void arrangeTeamsNicely() {
		int buffer = Config.wb_distanceBetweenCards;
		Point corner = new Point(buffer, buffer);
		int columnWidth = 0;
		Dimension dim;
		for (WhiteboardTeam wt : teams) {
			wt.setTopLeftCorner(corner.x, corner.y);
			dim = wt.getDimensions();
			columnWidth = Math.max(columnWidth, dim.width);
			corner.y += dim.height;
			if (corner.y > windowSize.height) {
				corner.setLocation(corner.x + columnWidth + buffer, buffer);
				wt.setTopLeftCorner(corner.x, corner.y);
				columnWidth = dim.width;
				corner.y += dim.height;
			}
			corner.y += buffer;
		}
	}

	final protected Graphics2D getGraphics2D() {
		return (Graphics2D)getGraphics();
	}

	protected void highlightSimilarCards(Entity entity, EntityStatus ts) {
		if (!flashMode) {
			flashMode = true;
			flashCounter = 0;
			final int MAX = 5;
			final List<WhiteboardCard> luckyWinners = new ArrayList<WhiteboardCard>();
			for (WhiteboardTeam wbt : teams) {
				WhiteboardCard luckyWinner = wbt.getMatchingCard(entity, ts);
				if (luckyWinner != null) {
					luckyWinners.add(luckyWinner);
				}
			}
			flashTimer = new Timer(100, new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					for (WhiteboardCard wbc : luckyWinners) {
						wbc.toggleFlashMode();
						repaint();
					}
					++flashCounter;
					if (flashCounter > MAX) {
						flashTimer.stop();
						flashMode = false;
					}
				}
			});
			flashTimer.setInitialDelay(0);
			flashTimer.start();
		}

	}
	
	//NOT for general use!!
	public void setTestMode() {
		testing = true;
	}

}
