package projman;

import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import projman.DiffWard.Status;

public class DartboardCard extends NameCard {

	private java.awt.geom.Point2D.Float proportionalDistanceFromCenter;
	private boolean insideCircle;
	private List<Arrow> arrows;
	private Point2D.Float velocity;

	private enum IntersectionStrategy {
		SIMPLE,
		STAGGERED,
		CLOSEST
	};

	public DartboardCard(Entity entity) {
		super(entity);
		proportionalDistanceFromCenter = new java.awt.geom.Point2D.Float();
		insideCircle = true;		
		arrows = new ArrayList<Arrow>();
		ghost = true;
		velocity = new Point2D.Float();
	}

	public void respondToResize(Point2D center, int radius) {
		centerOfGravity.setLocation(center.getX() + radius * proportionalDistanceFromCenter.x,
				center.getY() + radius * proportionalDistanceFromCenter.y);
		adjustPositionFromCenterOfGravity();
	}

	public void doneDragging(Point center, int radius) {
		ghost = false;
		adjustProportionFromCenter(center, radius);
	}

	private void adjustPositionFromCenterOfGravity() {
		int x = centerOfGravity.x - geometry.width/2;
		int y = centerOfGravity.y - geometry.height/2;
		geometry.setLocation(x,y);
	}

	private void adjustProportionFromCenter(Point center, int radius) {
		//calculate proportional distance from center;
		//use it to calculate movement when screen resizes.
		Point distanceFromCardToCenter = new Point(
				centerOfGravity.x - center.x,
				centerOfGravity.y - center.y);
		proportionalDistanceFromCenter.x = (float)distanceFromCardToCenter.x / radius;
		proportionalDistanceFromCenter.y = (float)distanceFromCardToCenter.y / radius;
	}

	public void draw(Graphics2D g, Point center, int radius, Map<Entity, FamilyWedge> fw,
			Map<Entity, DartboardCard> cardIndex) {
		if (!isInitialized()) {
			adjustProportionFromCenter(center, radius);
		}

		//Don't draw family connections if the card is outside the circle,
		//OR if the card is a "ghost".
		if (insideCircle && !ghost && !isMoving()) {
			//draw family connections
			if (diffWard != null) {
				drawDiffConnections(g, fw);
			} else {
				drawConnections(g, fw, cardIndex);
			}
		}
		
		if (isMoving()) {
			drag(velocity);
		}

		super.draw(g);

	}

	private void drawDiffConnections(Graphics2D g, Map<Entity, FamilyWedge> fw) {
		arrows.clear();
		Point p1, p2;
		p1 = new Point();
		p2 = new Point();
		Arrow arrow;
		if (getEntity() instanceof Project) {
			Project project = (Project)getEntity();
			DiffProject dp = diffWard.getDiffProject(project);

			//draw lines connecting this project to its resource-wedges
			Set<Resource> assignedResources = dp.getResources();
			for (Resource res : assignedResources) {
				FamilyWedge w = findMatchingWedge(res, fw);
				p1.setLocation(w.getFocalPoint());
				p2 = centerOfGravity;
				arrow = new DiffArrow(p1, p2, res, project, dp.getResourceStatus(res));
				arrows.add(arrow);
				arrow.draw(g, Config.db_homeTeachingArrowColor);
			}

			//draw lines connecting this project to its skill-wedges
			for (String f : dp.getAllAssociatedSkills()) {
				//draw as many arrows as there are required skills

				FamilyWedge wedge = findMatchingWedge(f, fw);
				if (wedge != null) {
					int oldCount = dp.getCountForSkill(f, Status.OLD);
					int newCount = dp.getCountForSkill(f, Status.NEW);
					int lesser = Math.min(oldCount, newCount);
					int greater = Math.max(oldCount, newCount);
					Status greaterStatus = (oldCount==greater) ? Status.OLD : Status.NEW;
					for (int i=0; i<lesser; ++i) {
						p1.setLocation(wedge.getPointAlongInnerRadius(i, greater));
						p2 = centerOfGravity;
						arrow = new DiffArrow(p1, p2, project, null, Status.SAME);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);
					}
					for (int i=lesser; i<greater; ++i) {
						p1.setLocation(wedge.getPointAlongInnerRadius(i, greater));
						p2 = centerOfGravity;
						arrow = new DiffArrow(p1, p2, project, null, greaterStatus);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);						
					}
				}
			}
		}

		if (getEntity() instanceof Resource) {
			Resource resource = (Resource)getEntity();
			DiffResource dr = diffWard.getDiffResource(resource);

			//draw lines connecting this resource to its project-wedges
			for (DiffProject dp : diffWard.getAllProjects()) {
				Status status = dp.getResourceStatus(resource);
				//Main.say("resource: " + resource.getName() + " / project: " + dp.getProject().getName() + " / status: " + status);
				if (status != null) {
					Project project = dp.getProject();
					FamilyWedge wedge = findMatchingWedge(project, fw);
					p1.setLocation(wedge.getFocalPoint());
					p2 = centerOfGravity;
					arrow = new DiffArrow(p1, p2, resource, project, status);
					arrows.add(arrow);
					arrow.draw(g, Config.db_homeTeachingArrowColor);
				}
			}

			//draw lines connecting this resource to its skill-wedges
			for (String k : dr.getSkills()) {
				FamilyWedge wedge = findMatchingWedge(k, fw);
				p1.setLocation(wedge.getFocalPoint());
				p2 = centerOfGravity;
				arrow = new DiffArrow(p1, p2, resource, null, dr.getSkillStatus(k));
				arrows.add(arrow);
				arrow.draw(g, Config.db_homeTeachingArrowColor);
			}
		}

		if (getEntity() instanceof Skill) {
			Skill skill = (Skill)getEntity();

			//draw lines connecting this skill to the wedges representing
			//the projects which require this skill
			for (DiffProject dp : diffWard.getAllProjects()) {
				Project proj = dp.getProject();
				FamilyWedge wedge = findMatchingWedge(proj, fw);
				if (wedge != null) {
					int oldCount = dp.getCountForSkill(skill.getName(), Status.OLD);
					int newCount = dp.getCountForSkill(skill.getName(), Status.NEW);
					int lesser = Math.min(oldCount, newCount);
					int greater = Math.max(oldCount, newCount);
					Status greaterStatus = (oldCount==greater) ? Status.OLD : Status.NEW;
					for (int i=0; i<lesser; ++i) {
						p1.setLocation(wedge.getPointAlongInnerRadius(i, greater));
						p2 = centerOfGravity;
						arrow = new DiffArrow(p1, p2, proj, skill, Status.SAME);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);
					}
					for (int i=lesser; i<greater; ++i) {
						p1.setLocation(wedge.getPointAlongInnerRadius(i, greater));
						p2 = centerOfGravity;
						arrow = new DiffArrow(p1, p2, proj, skill, greaterStatus);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);						
					}
				}
			}

			//draw lines connecting this skill to its resource wedges
			for (DiffResource dr : diffWard.getAllResources()) {
				//Main.say("resource: " + dr.getResource().getName());
				Status status = dr.getSkillStatus(skill.getName());
				if (status != null) {
					Resource res = dr.getResource();
					FamilyWedge wedge = findMatchingWedge(res, fw);
					p1.setLocation(wedge.getFocalPoint());
					p2 = centerOfGravity;
					arrow = new DiffArrow(p1, p2, skill, res, status);
					arrows.add(arrow);
					arrow.draw(g, Config.db_homeTeachingArrowColor);
				}
			}
		}


	}

	private void drawConnections(Graphics2D g, Map<Entity, FamilyWedge> fw,
			Map<Entity, DartboardCard> cardIndex) {
		//TODO There is room for optimization here.
		//There's no reason why we should recompute all the
		//connections and rebuild all the Arrow objects each
		//time we redraw the screen.  This is only necessary
		//when the ward changes.
		arrows.clear();
		Point p1, p2;
		p1 = new Point();
		p2 = new Point();
		Arrow arrow;
		if (getEntity() instanceof Project) {
			Project project = (Project)getEntity();
			//draw lines connecting this project to its resource-wedges
			Team assignedResources = project.getTeachers();
			if (assignedResources != null) {
				for (ResourceBase f : assignedResources.getAllTeachers()) {
					p1 = getOppositePoint(f, fw, cardIndex, IntersectionStrategy.CLOSEST, 0, 0);
					if (p1 != null) {
						p2 = lineRectangleIntersection(centerOfGravity, p1, geometry);
						arrow = new Arrow(p1, p2, f, project);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);
					}
				}
			}
			//draw lines connecting this project to its skill-wedges
			for (Skill f : project.getRequiredSkills()) {
				int numRequired = project.howManyNeeded(f);
				int numAssigned = project.howManyOfOurResourcesHaveThisSkill(f);
				for (int i=0; i<numRequired; ++i) {
					p1 = getOppositePoint(f, fw, cardIndex, IntersectionStrategy.STAGGERED, i, numRequired);
					if (p1 != null) {
						p2 = lineRectangleIntersection(centerOfGravity, p1, geometry);
						arrow = new Arrow(p2, p1, project, f);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);
						//draw bidirectional arrows if this project is assigned
						//a resource with the skill required.
						if (numAssigned > i) {
							arrow = new Arrow(p1, p2, f, project);
							arrows.add(arrow);
							arrow.draw(g, Config.db_homeTeachingArrowColor);
						}
					}
				}
			}
		}

		if (getEntity() instanceof Resource) {
			Resource resource = (Resource)getEntity();
			List<Team> teams = resource.getCompanionships();
			//draw lines connecting this resource to its project-wedges
			for (Team team : teams) {
				for (ProjectBase f : team.getAllTeachees()) {
					p1 = getOppositePoint(f, fw, cardIndex, IntersectionStrategy.SIMPLE, 0, 0);
					if (p1 != null) {
						p2 = lineRectangleIntersection(centerOfGravity, p1, geometry);
						arrow = new Arrow(p2, p1, resource, f);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);
					}
				}
			}
			//draw lines connecting this resource to its skill-wedges
			for (Skill f : resource.getSkillSet()) {
//				FamilyWedge wedge = fw.get(f);
//				if (wedge != null) {
//				p1.setLocation(wedge.getFocalPoint());
				p1 = getOppositePoint(f, fw, cardIndex, IntersectionStrategy.SIMPLE, 0, 0);
				if (p1 != null) {
					p2 = lineRectangleIntersection(centerOfGravity, p1, geometry);
					arrow = new Arrow(p2, p1, resource, f);
					arrows.add(arrow);
					arrow.draw(g, Config.db_homeTeachingArrowColor);
				}				
			}
		}
		if (getEntity() instanceof Skill) {
			Skill skill = (Skill)getEntity();
			List<Project> projects = skill.getMatchingProjects();
			//draw lines from this skill to its project-wedges
			for (Project f : projects) {
//				FamilyWedge wedge = fw.get(f);
//				if (wedge != null) {
				//check how many instances of this skill
				//are required by this project, and draw that
				//many arrows.
				int numRequired = f.howManyNeeded(skill);
				int numAssigned = f.howManyOfOurResourcesHaveThisSkill(skill);
				for (int i=0; i<numRequired; ++i) {
					p1 = getOppositePoint(f, fw, cardIndex, IntersectionStrategy.STAGGERED, i, numRequired);
					if (p1 != null) {
						//p1.setLocation(wedge.getPointAlongInnerRadius(i, numRequired));
						p2 = lineRectangleIntersection(centerOfGravity, p1, geometry);
						arrow = new Arrow(p1, p2, f, skill);
						arrows.add(arrow);
						arrow.draw(g, Config.db_homeTeachingArrowColor);
						//draw bidirectional arrows if a project is assigned
						//a resource possessing the skill required.
						if (numAssigned > i) {
							arrow = new Arrow(p2, p1, skill, f);
							arrows.add(arrow);
							arrow.draw(g, Config.db_homeTeachingArrowColor);
						}

					}
				}
			}
			//draw lines from this skill to its resource-wedges
			List<Resource> resources = skill.getMatchingResources();
			for (Resource f : resources) {
//				FamilyWedge wedge = fw.get(f);
//				if (wedge != null) {
//					p1.setLocation(wedge.getFocalPoint());
				p1 = getOppositePoint(f, fw, cardIndex, IntersectionStrategy.SIMPLE, 0, 0);
				if (p1 != null) {
					p2 = lineRectangleIntersection(centerOfGravity, p1, geometry);
					arrow = new Arrow(p1, p2, f, skill);
					arrows.add(arrow);
					arrow.draw(g, Config.db_homeTeachingArrowColor);
				}
			}
		}

	}

	private Point getOppositePoint(Entity otherEntity, Map<Entity, FamilyWedge> fw,
			Map<Entity, DartboardCard> cardIndex, IntersectionStrategy strategy, int i, int n) {
		Point result = null;
		DartboardCard otherCard = cardIndex.get(otherEntity);
		FamilyWedge wedge = fw.get(otherEntity);
		if (!(otherCard == null && wedge == null)) {
			if (Config.db_newStyle && otherCard != null) {
				result = lineRectangleIntersection(otherCard.centerOfGravity, centerOfGravity, otherCard.geometry);
			} else {
				result = new Point();
				switch (strategy) {
				case CLOSEST:
					result.setLocation(wedge.getPointClosestToAssignedProject((Project)getEntity()));
					break;
				case STAGGERED:
					result.setLocation(wedge.getPointAlongInnerRadius(i, n));
					break;
				case SIMPLE:
				default:
					result.setLocation(wedge.getFocalPoint());
				break;
				}
			}
		}
		return result;
	}

	private Point lineIntersection(Point p1, Point p2,
			Point p3, Point p4) {
		// adapted from:
		// http://local.wasp.uwa.edu.au/~pbourke/geometry/lineline2d/
		Point result = null;
		float denominator = (p4.y - p3.y) * (p2.x - p1.x) - (p4.x - p3.x) * (p2.y - p1.y);
		if (denominator != 0) {
			float numerator = (p4.x - p3.x) * (p1.y - p3.y) - (p4.y - p3.y) * (p1.x - p3.x);
			float ua = numerator / denominator;
			numerator = (p2.x - p1.x) * (p1.y - p3.y) - (p2.y - p1.y) * (p1.x - p3.x);
			float ub = numerator / denominator;
			if (ua > 0 && ua < 1 && ub > 0 && ub < 1) {
				result = new Point();
				result.x = p1.x + Math.round(ua * (p2.x - p1.x));
				result.y = p1.y + Math.round(ua * (p2.y - p1.y));
			}
		}
		return result;
	}

	private Point lineRectangleIntersection(Point p3, Point p4, Rectangle box) {
		//test each segment of the rectangle in turn
		Point result;
		//top edge
		Point p1 = new Point(box.getLocation());
		Point p2 = new Point(box.getLocation());
		p2.x += box.width;
		result = lineIntersection(p1, p2, p3, p4);
		if (result != null) return result;

		//right edge
		p1.x += box.width;
		p2.y += box.height;
		result = lineIntersection(p1, p2, p3, p4);
		if (result != null) return result;

		//bottom edge
		p1.y += box.height;
		p2.x = box.x;
		result = lineIntersection(p1, p2, p3, p4);
		if (result != null) return result;

		//left edge
		p1.x = box.x;
		p2.y = box.y;
		result = lineIntersection(p1, p2, p3, p4);

		return result;
	}

	/**
	 * NOTE: This method takes advantage of our assumption that we do NOT
	 * use gestures in DiffBoard. If that ever changes, this method will
	 * break.
	 * @param line
	 * @return
	 */
	public boolean checkGestureForIntersectionsWithArrows(Line2D.Float line) {
		boolean needToUpdateAssignments = false;
		for (Arrow arrow : arrows) {
			if (arrow.intersects(line)) {
				if (arrow.from.attemptToDisaffiliate(arrow.to)) {
					needToUpdateAssignments = true;
				}
			}
		}
		return needToUpdateAssignments;
	}

//	@Override
//	protected Color getCardColor() {
//	return Config.getColorForEntity(getEntity(), diffStatus);
//	}

	public boolean isInsideCircle() {
		return insideCircle;
	}

	public void setInsideCircle(boolean insideCircle) {
		this.insideCircle = insideCircle;
	}

	private FamilyWedge findMatchingWedge(Entity fam, Map<Entity, FamilyWedge> fw) {
		FamilyWedge result = null;
		for (Entity candidate : fw.keySet()) {
			if (fam.equals(candidate)) {
				result = fw.get(candidate);
				break;
			}
		}
		return result;
	}

	private FamilyWedge findMatchingWedge(String fam, Map<Entity, FamilyWedge> fw) {
		FamilyWedge result = null;
		for (Entity candidate : fw.keySet()) {
			if (fam.equals(candidate.getName())) {
				result = fw.get(candidate);
				break;
			}
		}
		return result;
	}
	
	public void setVelocity(float run, float rise) {
		velocity.setLocation(run, rise);
	}
	
	private boolean isMoving() {
		return (velocity.x != 0 || velocity.y != 0);
	}
	

}
