package projman;


import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JPanel;
import javax.swing.Timer;
import javax.swing.event.MouseInputAdapter;

import projman.Ward.WardListener;

public class DartboardTab extends JPanel implements WardListener {
	private Ward ward;
	protected Map<Entity, FamilyWedge> wedges;
	protected List<FamilyWedge> sequentialWedges;
	protected List<DartboardCard> cards;
	protected Map<Entity, DartboardCard> cardIndex;
	protected Set<Entity> excludeFromRing;
	private static final long serialVersionUID = 42L;
	protected RadialCache cache;
	protected Dimension windowSize;
	protected boolean dirty;
	protected float innerRadius, outerRadius;
	protected float[] radii;
	protected Point center;
	protected int spinFactor;
	protected boolean spinning;
	protected boolean drawRingBorders;
	protected Gesture gesture;
	private FamilyWedge closestWedge;
	private FamilyWedge wedge1ToTheLeft;
	private FamilyWedge wedge1ToTheRight;
	private FamilyWedge wedge2ToTheLeft;
	private FamilyWedge wedge2ToTheRight;
	private float closestWedgeLength;
	protected boolean needToZoomWedges;
	private boolean needToResetWedges;
	private float growthRatio;
	private float animationRatio;
	//private int flyingCardStep;
	private boolean currentlyAnimating;
	private Timer timer;//moving this to class level to satisfy compiler

	private enum AnimationStyle {
		GROW,
		SHRINK
	}

	protected DartboardTab() {
		super();
		wedges = new HashMap<Entity, FamilyWedge>();
		cardIndex = new HashMap<Entity, DartboardCard>();
		sequentialWedges = new ArrayList<FamilyWedge>();
		cards = new LinkedList<DartboardCard>();
		excludeFromRing = new HashSet<Entity>();
		cache = new RadialCache();
		radii = new float[Config.db_totalNumberOfRadii];
		spinFactor = 0;
		spinning = false;
		addKeyListener(new KeystrokeHandler());
		dirty = false;
		drawRingBorders = false;
		needToZoomWedges = false;
		needToResetWedges = false;
		gesture = null;
		closestWedge = null;
		nullifyNeighborWedges();
		closestWedgeLength = Float.MAX_VALUE;
		growthRatio = 0;
		currentlyAnimating = false;
	}

	public DartboardTab(Ward ward) {
		this();
		this.ward = ward;
		this.ward.addWardListener(this);
		DartboardMouseHandler mouse = new DartboardMouseHandler();
		addMouseListener(mouse);
		addMouseMotionListener(mouse);
		init();
	}

	protected void init() {
		recalibrateWedges();
		resize();
		resizeWedges();		
	}

	protected void resize() {
		windowSize = getSize();
		//Main.say("inside resize(): windowSize="+windowSize);
		//We want the circle to be big enough to fit comfortably inside the window,
		//but not so small that it folds in on itself.
		int totalWedgeThickness = Config.db_regularWedgeThickness
		+ Config.db_distanceBetweenRegularAndProgressWedge
		+ Config.db_progressWedgeThickness;
		outerRadius = Math.max(
				Math.min(windowSize.width, windowSize.height)/2 - Config.db_distanceBetweenWindowEdgeAndOuterRadius,
				totalWedgeThickness);
		innerRadius = outerRadius - totalWedgeThickness;

		radii[0] = innerRadius;
		radii[1] = radii[0] + Config.db_regularWedgeThickness;
		radii[2] = radii[1] + Config.db_distanceBetweenRegularAndProgressWedge;
		int distanceBetweenSegments = Config.db_progressWedgeThickness / Config.db_numProgressSegments;
		for (int i=3; i<Config.db_totalNumberOfRadii; ++i) {
			radii[i] = radii[i-1] + distanceBetweenSegments;
		}

		//Done resizing wedges, now adjust card position
		center = new Point(windowSize.width/2, windowSize.height/2);
		cache.setRadii(radii, center);
		for (DartboardCard c : cards) {
			c.respondToResize(center, (int)innerRadius);
		}
		dirty = true;
	}

	protected void resizeWedges() {
		float swellFactor;
		for (FamilyWedge fw : wedges.values()) {
			swellFactor = 0;
			if (needToResetWedges) {
				fw.reset();
			}
			if (fw == closestWedge) {
				swellFactor = 1+growthRatio;
			}
			if (fw == wedge1ToTheLeft || fw == wedge1ToTheRight) {
				swellFactor = 1+0.66f*growthRatio;
			}
			if (fw == wedge2ToTheLeft || fw == wedge2ToTheRight) {
				swellFactor = 1+0.33f*growthRatio;
			}
			fw.buildVertices(cache, windowSize, spinFactor, swellFactor, innerRadius);
		}
		if (sequentialWedges.size() == 0) {
			drawRingBorders = true;
		} else {
			drawRingBorders = false;
		}
		needToResetWedges = false;
	}

	public void wardChanged() {
		recalibrateWedges();
	}

	//helper
	protected void add(FamilyWedge fw) {
		sequentialWedges.add(fw);
		wedges.put(fw.getEntity(), fw);
	}

	//helper
	protected void deleteAllWedges() {
		wedges.clear();
		sequentialWedges.clear();	
	}

	protected void recalibrateWedges() {
		recalibrateWedgesWithoutDrawing();
		dirty = true;
		if (isShowing()) repaint();
	}

	protected void recalibrateWedgesWithoutDrawing() {
		float start = RadialCache.RESOLUTION/4f;
		deleteAllWedges();
		float numUnitsPerWedge;
		int totalPriority = 0;
		float partitionSize = determinePartitionSize();
		float projectPartitionSize = partitionSize;
		float resourcePartitionSize = partitionSize;
		float skillPartitionSize = partitionSize;

		//We start by figuring out the TOTAL priority of all projects
		for (Project p : ward.getAllProjects()) {
			if (excludeFromRing.contains(p)) continue;
			totalPriority += p.getPriority().ordinal();
		}

		//bookkeeping
		int visibleResources = 0;
		for (Resource f : ward.getAllResources()) {
			if (!excludeFromRing.contains(f)) ++visibleResources;
		}
		int visibleSkills = 0;
		for (Skill f : ward.getAllSkills()) {
			if (!excludeFromRing.contains(f)) ++visibleSkills;
		}

		//only used for PM frills; you could wrap this in an
		//if-statement if you cared enough to.
		int visibleProjects = 0;
		for (Project f : ward.getAllProjects()) {
			if (!excludeFromRing.contains(f)) ++visibleProjects;
		}

		//This next section of code calculates how long each wedge should be.

		//Next, we figure out the RELATIVE priority of each project,
		//and size its wedge accordingly.
		for (Project f : ward.getAllProjects()) {
			if (excludeFromRing.contains(f)) continue;
			float relativePriority = f.getPriority().ordinal() / (float)totalPriority;
			if (Config.db_frills) {
				numUnitsPerWedge = projectPartitionSize * relativePriority;
			} else {
				numUnitsPerWedge = projectPartitionSize / visibleProjects;
			}
			FamilyWedge fw = new FamilyWedge(f, start, numUnitsPerWedge);
			start += numUnitsPerWedge;
			add(fw);
		}

		//Then, we give all visible resources equally-sized wedges.
		numUnitsPerWedge = resourcePartitionSize / visibleResources;
		for (Resource f : ward.getAllResources()) {
			if (excludeFromRing.contains(f)) continue;
			FamilyWedge fw;
			fw = new FamilyWedge(f, start, numUnitsPerWedge);
			start += numUnitsPerWedge;
			add(fw);
		}

		//Same thing for all visible skills.  Divide up the space
		//available into equally-sized spans.
		numUnitsPerWedge = skillPartitionSize / visibleSkills;
		for (Skill f : ward.getAllSkills()) {
			if (excludeFromRing.contains(f)) continue;
			FamilyWedge fw;
			fw = new FamilyWedge(f, start, numUnitsPerWedge);
			start += numUnitsPerWedge;
			add(fw);
		}

		//if the family no longer exits, remove the card.
		List<DartboardCard> condemned = new ArrayList<DartboardCard>();
		for (DartboardCard pc : cards) {
			if (!(ward.containsEntity(pc.getEntity()))) {
				condemned.add(pc);
			}
		}
		for (DartboardCard pc : condemned) {
			cards.remove(pc);
		}

		//deactivate those wedges who have corresponding 
		//cards inside the circle
		for (DartboardCard card : cards) {
			if (excludeFromRing.contains(card.getEntity())) continue;
			wedges.get(card.getEntity()).setActive(false);
		}

	}

	@Override
	public void paintComponent(Graphics g1) {
		requestFocusInWindow();
		Graphics2D g = (Graphics2D)g1;
		if (!(windowSize.equals(getSize()))) {
			resize();
		}
		if (needToZoomWedges) {
			zoomWedges();
			needToZoomWedges = false;
		}
		if (dirty) {
			resizeWedges();
			dirty = false;
		}

		//turn on antialiasing, if desired.
		if (Config.antialias) {
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		} else {
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
		}

		//set the line thickness
		g.setStroke(Config.regularLine);

		//Clear screen to background color
		g.setColor(Config.backgroundColor);
		g.fillRect(0,0,windowSize.width, windowSize.height);

		//fill the background of the circle with color
		g.setColor(Config.circleBackgroundColor);
		g.fill(new Ellipse2D.Float(center.x-innerRadius, center.y-innerRadius, innerRadius*2, innerRadius*2));

		//draw all the wedges
		for (FamilyWedge fw : wedges.values()) {
			fw.draw(g);
		}

		//draw the cards
		for (DartboardCard c : cards) {
			c.draw(g, center, (int)innerRadius, wedges, cardIndex);
		}

		//draw a ring around the wedges if necessary
		if (drawRingBorders) {
			g.setColor(Config.db_ringBorderColor);
			g.draw(new Ellipse2D.Float(center.x-innerRadius, center.y-innerRadius, innerRadius*2, innerRadius*2));
			g.draw(new Ellipse2D.Float(center.x-outerRadius, center.y-outerRadius, outerRadius*2, outerRadius*2));
		}

		if (gesture != null) {
			g.setColor(Color.RED);
			gesture.draw(g);
		}
		//Main.say("");
	}

	protected class DartboardMouseHandler extends MouseInputAdapter implements Serializable {
		private boolean dragging = false;
		private Point prevMousePos, mousePos;
		protected DartboardCard selectedCard = null;
		private boolean rubicon = true;//designates whether a newly-created card has left the "red circle" zone yet or not.
		//If not, don't draw the red circles. If so, draw them.
		private final int DISTANCE_THRESHOLD = 10000;
		private final float MAX_WEDGE_LENGTH = RadialCache.RESOLUTION/12f;
		private static final long serialVersionUID = 42L;

		@Override
		public void mousePressed(MouseEvent e) {
			//process no mouse events if we're animating.
			if (currentlyAnimating) return;

			mousePos = e.getPoint();
			prevMousePos = mousePos;
			if (e.getButton() == 1) {
				//first check if the user clicked on a card...
				//(start checking at the *end* of the card list
				//so that we preferentially select those on top.)
				for (int i=cards.size()-1; i>=0; --i) {
					DartboardCard c = cards.get(i);
					if (c.containsPoint(mousePos)) {
						selectedCard = c;
						dragging = true;
						selectedCard.setPrevPosition(selectedCard.getPosition());
						break;
					}
				}
				//did he click inside or outside the "inner radius" of the ring?
				//if outside, check to see if he clicked on a wedge.
				if (mousePos.distance(windowSize.width/2, windowSize.height/2) > innerRadius) {
					for (FamilyWedge fw : wedges.values()) {
						if (fw.containsPoint(mousePos)) {
							selectedCard = new DartboardCard(fw.getEntity());
							selectedCard.setPosition(mousePos);
							dragging = true;
							rubicon = false;
							cards.add(selectedCard);
							fw.setActive(false);
							break;
						}
					}
				} else {
					//if the use clicked inside the circle but did not
					//click on a card, initiate a mouse gesture.
					if (!dragging) {
						gesture = new Gesture(mousePos);
					}
				}
			} else {
				//if the user clicked the 2nd or 3rd mouse button,
				//just assume we're spinning.
				spinning = true;
			}
			//if we're dragging a card, move the card to the
			//end of the list so that it "floats" above the
			//other cards when rendered.
			if (dragging && (selectedCard != null)) {
				cards.remove(selectedCard);
				cards.add(selectedCard);
			}
		}

		@Override
		public void mouseReleased(MouseEvent e) {
			//process no mouse events if we're animating.
			if (currentlyAnimating) return;

			mousePos = e.getPoint();
			spinning = false;
			rubicon = true;
			drawRingBorders = false;
			if (dragging) {
				dragging = false;
				int distanceFromCenter = (int)mousePos.distance(windowSize.width/2, windowSize.height/2);
				if (selectedCard != null) {
					if (distanceFromCenter > outerRadius) {
						//just dock it outside, and remove
						//the wedge from the circle.
						selectedCard.setInsideCircle(false);
						selectedCard.doneDragging(center, (int)innerRadius);
						initiateAnimation(wedges.get(selectedCard.getEntity()), AnimationStyle.SHRINK);
					} else {
						//the card was dropped inside the circle
						if (distanceFromCenter < innerRadius) {
							//check: did we drop this card onto an existing card?
							if (!checkForOverlappingCards(selectedCard)) {
								selectedCard.doneDragging(center, (int)innerRadius);
								selectedCard.setInsideCircle(true);
								if (Config.db_newStyle) {
									initiateAnimation(wedges.get(selectedCard.getEntity()), AnimationStyle.SHRINK);
								} else {
									if (excludeFromRing.remove(selectedCard.getEntity())) {
										recalibrateWedgesWithoutDrawing();
										initiateAnimation(wedges.get(selectedCard.getEntity()), AnimationStyle.GROW);
									}
									wedges.get(selectedCard.getEntity()).setActive(false);
								}
							} else {
								//the cards linked up, now handle the aftermath.
								if (Config.db_newStyle && excludeFromRing.contains(selectedCard.getEntity())) {
									selectedCard.setPosition(selectedCard.getPrevPosition());
								} else {
									cards.remove(selectedCard);
									cardIndex.remove(selectedCard.getEntity());
								}
								ward.assignmentsChanged();
							}
						} else {
							//the card was dropped in between the inner and outer radii.
							//put it back in the ring if it's not there.
							restoreWedge(selectedCard);
//							if (excludeFromRing.remove(selectedCard.getEntity())) {
//								recalibrateWedgesWithoutDrawing();
//								initiateAnimation(wedges.get(selectedCard.getEntity()), AnimationStyle.GROW);
//							}
//							wedges.get(selectedCard.getEntity()).setActive(true);
//							cards.remove(selectedCard);
//							cardIndex.remove(selectedCard.getEntity());
						}
					}
					selectedCard = null;

					//reset wedges; done zooming.
					if (closestWedge != null) {
						closestWedge = null;
						nullifyNeighborWedges();
						needToResetWedges = true;
						dirty = true;
					}

				}
				repaint();
			}
			if (gesture != null) {
				boolean needToUpdateAssignments = false;
				Line2D.Float line = gesture.getLine();
				if (line != null) {
					for (DartboardCard c : cards) {
						if (c.checkGestureForIntersectionsWithArrows(line)) {
							needToUpdateAssignments = true;
						}
					}
				}
				gesture = null;
				if (needToUpdateAssignments) {
					ward.assignmentsChanged();
				} else {
					repaint();//remove gesture from screen
				}
			}
		}

		@Override
		public void mouseDragged(MouseEvent e) {
			//process no mouse events if we're animating.
			if (currentlyAnimating) return;

			mousePos = e.getPoint();			
			Point delta = new Point(mousePos.x-prevMousePos.x,
					mousePos.y-prevMousePos.y);
			if (dragging) {
				int distanceFromCenter = (int)mousePos.distance(windowSize.width/2, windowSize.height/2);
				if (selectedCard != null) {
					selectedCard.drag(delta);
					//Determine whether we draw the red rings or not, based on whether the user
					//has crossed the "rubicon" or not.
					if (distanceFromCenter > innerRadius && distanceFromCenter < outerRadius) {
						if (rubicon) {
							drawRingBorders = true;
						}
					} else {
						drawRingBorders = false;
						rubicon = true;
						//okay, the rubicon's been crossed. Now are we inside or outside the ring?
						//If inside the circle, then make it so the arrows get drawn to the
						//selected card, NOT to the wedge.
						if (Config.db_newStyle) {
							if (distanceFromCenter < innerRadius) {
								cardIndex.put(selectedCard.getEntity(), selectedCard);
							}
							if (distanceFromCenter > outerRadius) {
								cardIndex.remove(selectedCard.getEntity());
							}
						}
					}
				}
			}
			if (spinning) {
				//translate the user's mouse movement into
				//a direction of rotation for the circle.
				int absX = Math.abs(delta.x);
				int absY = Math.abs(delta.y);
				if (mousePos.x > center.x) {
					if (mousePos.y > center.y) {
						if (absX > absY) spinFactor -= delta.x;
						else spinFactor += delta.y;
					} else {
						if (absX > absY) spinFactor += delta.x;
						else spinFactor += delta.y;
					}
				} else {
					if (mousePos.y > center.y) {
						if (absX > absY) spinFactor -= delta.x;
						else spinFactor -= delta.y;
					} else {
						if (absX > absY) spinFactor += delta.x;
						else spinFactor -= delta.y;
					}					
				}
				spinFactor %= RadialCache.RESOLUTION;
				dirty = true;
			}
			if (gesture != null) {
				gesture.addPoint(mousePos);
			}
			prevMousePos = mousePos;
			repaint();
		}

		@Override
		public void mouseMoved(MouseEvent e) {
			//process no mouse events if we're animating.
			if (currentlyAnimating) return;

			//This method has two purposes. One is to manage the appearance of tool tips.
			//The other is to manage the fish-eye zooming of the icons.
			try {
				mousePos = e.getPoint();
				//float distanceFromCenter = (float)mousePos.distance(windowSize.width/2, windowSize.height/2);
				FamilyWedge tempClosestWedge = null;
				int minDist = Integer.MAX_VALUE;
				String tooltip = null;
				for (FamilyWedge fw : wedges.values()) {
					int dist = (int)(mousePos.distanceSq(fw.getFocalPoint()));
					if (dist < minDist) {
						tempClosestWedge = fw;					
						minDist = dist;
					}
					String tooltipTmp = fw.getTooltipText(mousePos);
					if (tooltipTmp != null) {
						tooltip = tooltipTmp;
					}
				}
				//TODO should we keep the && clause?
				if (minDist < DISTANCE_THRESHOLD /*&& distanceFromCenter > innerRadius*/) {
					closestWedge = tempClosestWedge;
					float wedgeLength = closestWedge.getLength();
					float ideal = closestWedge.getIdealLength();
					growthRatio = (DISTANCE_THRESHOLD - minDist) / (float)DISTANCE_THRESHOLD;
					if (wedgeLength < MAX_WEDGE_LENGTH) {
						closestWedgeLength = ideal + growthRatio * (MAX_WEDGE_LENGTH-ideal);
					} else {
						closestWedgeLength = ideal;
					}
					needToZoomWedges = true;
					repaint();
				} else {
					//reset wedges; done zooming.
					if (closestWedge != null) {
						closestWedge = null;
						nullifyNeighborWedges();
						needToResetWedges = true;
						dirty = true;
						repaint();
					}
				}
				DartboardTab.this.setToolTipText(tooltip);
			} catch (Exception npe) {
				//do nothing; these mouse move events come fast
				//enough that if one chokes, we can skip it.
			}
		}
	}
	
	private void restoreWedge(DartboardCard dbc) {
		//the card was dropped in between the inner and outer radii.
		//put it back in the ring if it's not there.
		if (excludeFromRing.remove(dbc.getEntity())) {
			recalibrateWedgesWithoutDrawing();
			initiateAnimation(wedges.get(dbc.getEntity()), AnimationStyle.GROW);
		}
		wedges.get(dbc.getEntity()).setActive(true);
		cards.remove(dbc);
		cardIndex.remove(dbc.getEntity());
	}

	protected void zoomWedges() {
		int numWedges = sequentialWedges.size();

		//throw out the trivial cases (only one wedge, or
		//user has turned off zooming in configuration).
		if (numWedges <= 1 || !Config.db_zoom) return;

		float difference, delta;
		List<FamilyWedge> reorderedWedges = new ArrayList<FamilyWedge>();
		nullifyNeighborWedges();		
		//reset everything to its 'normal' size.
		for (FamilyWedge fw : sequentialWedges) {
			fw.reset();
		}

		if (closestWedge != null) {
			difference = closestWedgeLength - closestWedge.getLength();
			boolean foundIt = false;
			for (int i=0; i<numWedges; ++i) {
				FamilyWedge fw = sequentialWedges.get(i);
				if (fw == closestWedge) {
					foundIt = true;
					fw.resetLength(closestWedgeLength);
					fw.adjustStartByDelta(-difference/2f);
					wedge1ToTheLeft = sequentialWedges.get((numWedges+i-1) % numWedges);//workaround for weird Java modulus
					wedge1ToTheRight = sequentialWedges.get((i+1) % numWedges);
					wedge2ToTheLeft = sequentialWedges.get((numWedges+i-2) % numWedges);
					wedge2ToTheRight = sequentialWedges.get((i+2) % numWedges);
					reorderedWedges.add(fw);
				} else {
					if (foundIt) reorderedWedges.add(fw);
				}
			}
			for (FamilyWedge fw : sequentialWedges) {
				if (fw == closestWedge) break;
				reorderedWedges.add(fw);
			}

			float leftNeighbor1Length = Math.max(wedge1ToTheLeft.getLength(), closestWedgeLength * 0.66f);
			float leftNeighbor1Difference = leftNeighbor1Length - wedge1ToTheLeft.getLength();
			float rightNeighbor1Length = Math.max(wedge1ToTheRight.getLength(), closestWedgeLength * 0.66f);
			float rightNeighbor1Difference = rightNeighbor1Length - wedge1ToTheRight.getLength();
			float leftNeighbor2Length = Math.max(wedge2ToTheLeft.getLength(), closestWedgeLength * 0.33f);
			float leftNeighbor2Difference = leftNeighbor2Length - wedge2ToTheLeft.getLength();
			float rightNeighbor2Length = Math.max(wedge2ToTheRight.getLength(), closestWedgeLength * 0.33f);
			float rightNeighbor2Difference = rightNeighbor2Length - wedge2ToTheRight.getLength();
			delta = (difference+leftNeighbor1Difference+rightNeighbor1Difference
					+leftNeighbor2Difference+rightNeighbor2Difference) / (numWedges-5); //'cuz we're zooming 5 wedges
			float start = closestWedge.getStart();
			for (FamilyWedge fw : reorderedWedges) {
				if (fw != closestWedge) {
					fw.resetStart(start);
					if (fw == wedge1ToTheLeft) fw.resetLength(leftNeighbor1Length);
					else if (fw == wedge1ToTheRight) fw.resetLength(rightNeighbor1Length);
					else if (fw == wedge2ToTheLeft) fw.resetLength(leftNeighbor2Length);
					else if (fw == wedge2ToTheRight) fw.resetLength(rightNeighbor2Length);
					else fw.adjustLengthByDelta(-delta);
				}
				start += fw.getLength();
			}
		}
		dirty = true;
	}

	//TODO this method is overly complex.  Is there a way we can make it work
	//without defining two separate Timer classes?
	protected void initiateAnimation(final FamilyWedge animatedWedge, final AnimationStyle style) {
		//Kick out the trivial case.
		if (animatedWedge == null) return;

		if (Config.db_animate) {
			int delay = Config.db_millisecondsBetweenWedgeAnimationFrames;
			currentlyAnimating = true;
			final List<FamilyWedge> wedgesToTweak = buildListOfWedgesToAnimate(animatedWedge);
			if (style == AnimationStyle.SHRINK) {
				//TODO maybe make increment adjustable, based on the speed of the CPU?
				animationRatio = 1f;
				final float increment = -0.1f;
				timer = new Timer(delay, new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						animationRatio += increment;
						animateWedges(animatedWedge, wedgesToTweak);
						if (animationRatio < 0.111f) {
							timer.stop();
							if (excludeFromRing.add(animatedWedge.getEntity())) {
								recalibrateWedges();
								//homeOnTheRing(animatedWedge.getEntity());//TESTING
							}
							currentlyAnimating = false;
						}
					}
				});
				
			} else {	//AnimationStyle.GROW
				
				animationRatio = 0f;
				final float increment = 0.1f;
				timer = new Timer(delay, new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						animationRatio += increment;
						animateWedges(animatedWedge, wedgesToTweak);
						if (animationRatio > 0.89f) {
							timer.stop();
							recalibrateWedges();
							currentlyAnimating = false;
						}
					}
				});

			}
			timer.setInitialDelay(0);
			timer.start();			
		} else {
			if (style == AnimationStyle.SHRINK) {
				excludeFromRing.add(animatedWedge.getEntity());
			}
			recalibrateWedges();
		}
	}

	protected List<FamilyWedge> buildListOfWedgesToAnimate(FamilyWedge animatedWedge) {
		List<FamilyWedge> wedgesToTweak = new ArrayList<FamilyWedge>();
		//TODO there MUST be a better way to do this!
		if (animatedWedge.getEntity() instanceof Project) {
			for (Project p : ward.getAllProjects()) {
				if (!(excludeFromRing.contains(p))) {
					wedgesToTweak.add(wedges.get(p));
				}
			}
		}
		if (animatedWedge.getEntity() instanceof Resource) {
			for (Resource p : ward.getAllResources()) {
				if (!(excludeFromRing.contains(p))) {
					wedgesToTweak.add(wedges.get(p));
				}
			}
		}
		if (animatedWedge.getEntity() instanceof Skill) {
			for (Skill p : ward.getAllSkills()) {
				if (!(excludeFromRing.contains(p))) {
					wedgesToTweak.add(wedges.get(p));
				}
			}
		}
		return wedgesToTweak;
	}

	private void animateWedges(FamilyWedge animatedWedge, List<FamilyWedge> wedgesToTweak) {
		float newLengthForAnimatedWedge;
		float delta;
		float start;

		//reset everything to its 'normal' size.
		for (FamilyWedge fw : wedgesToTweak) {
			fw.reset();
		}

		newLengthForAnimatedWedge = animatedWedge.getLength() * animationRatio;
		delta = (animatedWedge.getLength() - newLengthForAnimatedWedge) / (wedgesToTweak.size()-1);
		start = wedgesToTweak.get(0).getStart();
		//Main.say("length: " + newLengthForAnimatedWedge);
		for (FamilyWedge fw : wedgesToTweak) {
			fw.resetStart(start);
			if (fw == animatedWedge) {
				fw.resetLength(newLengthForAnimatedWedge);
			} else {
				fw.adjustLengthByDelta(delta);
			}
			start += fw.getLength();
		}
		dirty=true;
		repaint();
	}


	//TODO see if you can make this a little more intelligent.
	//Why should a category get 1/3 of the circle if it only has
	//2 or 3 wedges, when another category has to cram all of its
	//10 wedges into the same-sized space?
	protected float determinePartitionSize() {
		//if any one of the 3 groups has no visible members, then the
		//remaining 2 groups get 180 degrees each.  Otherwise each
		//gets 120 degrees.
		//NOTE: this method probably won't win any awards for
		//elegance, but it gets the job done.
		boolean resourcesAreVisible = false;
		boolean projectsAreVisible = false;
		boolean skillsAreVisible = false;
		float result;
		for (Resource p : ward.getAllResources()) {
			if (excludeFromRing.contains(p)) continue;
			resourcesAreVisible = true;
			break;
		}
		for (Skill p : ward.getAllSkills()) {
			if (excludeFromRing.contains(p)) continue;
			skillsAreVisible = true;
			break;
		}
		for (Project p : ward.getAllProjects()) {
			if (excludeFromRing.contains(p)) continue;
			projectsAreVisible = true;
			break;
		}
		if (resourcesAreVisible && skillsAreVisible && projectsAreVisible) {
			result = RadialCache.RESOLUTION/3f;
		} else {
			result = RadialCache.RESOLUTION/2f;
		}
		return result;
	}

	protected boolean checkForOverlappingCards(DartboardCard selectedCard) {
		boolean match = false;
		for (DartboardCard card : cards) {
			if (card.intersects(selectedCard)) {
				Entity cardEntity = card.getEntity();
				Entity entity = selectedCard.getEntity();
				if (entity.equals(cardEntity)) continue;
				match = cardEntity.attemptToAffiliate(entity);
				break;
			}
		}
		return match;
	}

	//lame helper method
	protected void nullifyNeighborWedges() {
		wedge1ToTheLeft = null;
		wedge1ToTheRight = null;
		wedge2ToTheLeft = null;
		wedge2ToTheRight = null;
	}

	//TODO eventually, just make this a "shortcut" for a toolbar icon.
	//Then we won't need a separate KeyListener.
	private class KeystrokeHandler extends KeyAdapter implements Serializable {
		@Override
		public void keyReleased(KeyEvent e) {
			if (e.getKeyCode() == KeyEvent.VK_SPACE) {
				clearBoard();
			}
		}
	}

	//TODO make this fancier, so that the cards fly back
	//to their respective wedges. Let the 'excluded'
	//cards return home first, so that the ring goes back
	//to "full size". Then let the cards inside the ring
	//go back to their places too.
	private void clearBoard() {
		
		List<DartboardCard> orphans = new ArrayList<DartboardCard>();
		for (Entity entity : excludeFromRing) {
			for (DartboardCard c : cards) {
				if (c.getEntity() == entity) {
					orphans.add(c);
					break;
				}
			}
		}
		currentlyAnimating = true;
		flyHome(orphans, true);	
		flyHome(cards, false);
		cards.clear();
		repaint();
		currentlyAnimating = false;
	}
	
	private void flyHome(List<DartboardCard> movingCards, boolean restore) {
		final int steps = Config.db_flyingCardSteps;
		if (movingCards.isEmpty() || !Config.db_animate) return;
		resizeWedges();
		for (DartboardCard c : movingCards) {
			Point2D.Float home = homeOnTheRing(c.getEntity());
			float run = (home.x - c.getPosition().x) / steps;
			float rise = (home.y - c.getPosition().y) / steps;
			c.setVelocity(run, rise);
		}
		for (int i=0; i<steps; ++i) {
			Main.pause(Config.db_millisecondsBetweenWedgeAnimationFrames);
			//FIXME paintImmediately() is a hack function. Replace it with
			//an elegant timer or something.
			paintImmediately(0,0,windowSize.width, windowSize.height);
		}
		for (DartboardCard dbc : movingCards) {
			dbc.setVelocity(0, 0);
			//FIXME - make animation work correctly with multiple cards.
			if (restore) {
				restoreWedge(dbc);
			} else {
				wedges.get(dbc.getEntity()).setActive(true);
			}
		}
		
	}
	
	//TODO keep this method around for later use, when you incorporate
	//animation into the "clearBoard" operation.
	private Point2D.Float homeOnTheRing(Entity entity) {
		Point2D.Float result = null;
		if (excludeFromRing.contains(entity)) {
			//find place between wedges
			Integer before = null;
			Integer after = null;
			FamilyWedge current = null;
			int comparison = 0;
			int size = sequentialWedges.size();
			for (int i=0; i<size; ++i) {
				current = sequentialWedges.get(i);
				comparison = entity.compareTo(current.getEntity());
				if (comparison == 0) {
					//not same type
					continue;
				}
				if (comparison < 0) {
					after = i;
					break;
				}
				if (comparison > 0) {
					before = i;
				}
			}
			if (after == null) {
				after = (size + before + 1) % size;
			}
			if (before == null) {
				before = (size + after - 1) % size;
			}
			result = sequentialWedges.get(after).getStartPoint();
		} else {
			result = wedges.get(entity).getFocalPoint();
		}
		return result;
	}

}
