package projman;


import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.List;

public class DiffDartboardTab extends DartboardTab {

	private DiffWard diffWard;

	public DiffDartboardTab(DiffWard ward) {
		super();
		this.diffWard = ward;
		DiffDartboardMouseHandler mouse = new DiffDartboardMouseHandler();
		addMouseListener(mouse);
		addMouseMotionListener(mouse);
		init();
	}

	@Override
	public void wardChanged() {
		diffWard.recomputeDifferences();
		recalibrateWedges();
	}

	@Override
	protected void recalibrateWedgesWithoutDrawing() {
		float start = RadialCache.RESOLUTION/4f;
		deleteAllWedges();
		float numUnitsPerWedge;
		float partitionSize = determinePartitionSize();
		float projectPartitionSize = partitionSize;
		float resourcePartitionSize = partitionSize;
		float skillPartitionSize = partitionSize;

		//bookkeeping
		int visibleResources = 0;
		for (DiffResource df : diffWard.getAllResources()) {
			Resource f = df.getResource();
			if (!excludeFromRing.contains(f)) ++visibleResources;
		}
		int visibleProjects = 0;
		for (DiffProject df : diffWard.getAllProjects()) {
			Project f = df.getProject();
			if (!excludeFromRing.contains(f)) ++visibleProjects;
		}
		int visibleSkills = 0;
		for (Skill f : diffWard.getAllSkills()) {
			if (!excludeFromRing.contains(f)) ++visibleSkills;
		}

		//This next section of code calculates how long each wedge should be.

		//Next, we figure out the RELATIVE priority of each project,
		//and size its wedge accordingly.
		numUnitsPerWedge = projectPartitionSize / visibleProjects;
		for (DiffProject df : diffWard.getAllProjects()) {
			Project f = df.getProject();
			if (excludeFromRing.contains(f)) continue;
			FamilyWedge fw = new FamilyWedge(f, start, numUnitsPerWedge, df.getProjectStatus());
			start += numUnitsPerWedge;
			add(fw);
		}

		//Then, we give all visible resources equally-sized wedges.
		numUnitsPerWedge = resourcePartitionSize / visibleResources;
		for (DiffResource dr : diffWard.getAllResources()) {
			Resource f = dr.getResource();
			if (excludeFromRing.contains(f)) continue;
			FamilyWedge fw;
			fw = new FamilyWedge(f, start, numUnitsPerWedge, dr.getResourceStatus());
			start += numUnitsPerWedge;
			add(fw);
		}

		//Same thing for all visible skills.  Divide up the space
		//available into equally-sized spans.
		numUnitsPerWedge = skillPartitionSize / visibleSkills;
		for (Skill f : diffWard.getAllSkills()) {
			if (excludeFromRing.contains(f)) continue;
			FamilyWedge fw;
			fw = new FamilyWedge(f, start, numUnitsPerWedge, diffWard.getStatus(f));
			start += numUnitsPerWedge;
			add(fw);
		}

		//deactivate those wedges who have corresponding 
		//cards inside the circle
		for (DartboardCard card : cards) {
			if (excludeFromRing.contains(card.getEntity())) continue;
			wedges.get(card.getEntity()).setActive(false);
		}

	}

	protected class DiffDartboardMouseHandler extends DartboardMouseHandler {

		@Override
		public void mousePressed(MouseEvent e) {
			super.mousePressed(e);
			gesture = null;
			if (selectedCard != null) {
				selectedCard.setDiffWard(diffWard);
			}
		}
	}


	@Override 
	protected List<FamilyWedge> buildListOfWedgesToAnimate(FamilyWedge animatedWedge) {
		List<FamilyWedge> wedgesToTweak = new ArrayList<FamilyWedge>();
		//TODO there MUST be a better way to do this!
		if (animatedWedge.getEntity() instanceof Project) {
			for (DiffProject dp : diffWard.getAllProjects()) {
				Project p = dp.getProject();
				if (!(excludeFromRing.contains(p))) {
					wedgesToTweak.add(wedges.get(p));
				}
			}
		}
		if (animatedWedge.getEntity() instanceof Resource) {
			for (DiffResource dp : diffWard.getAllResources()) {
				Resource p = dp.getResource();
				if (!(excludeFromRing.contains(p))) {
					wedgesToTweak.add(wedges.get(p));
				}
			}
		}
		if (animatedWedge.getEntity() instanceof Skill) {
			for (Skill p : diffWard.getAllSkills()) {
				if (!(excludeFromRing.contains(p))) {
					wedgesToTweak.add(wedges.get(p));
				}
			}
		}
		return wedgesToTweak;
	}


	//TODO see if you can make this a little more intelligent.
	//Why should a category get 1/3 of the circle if it only has
	//2 or 3 wedges, when another category has to cram all of its
	//10 wedges into the same-sized space?
	@Override
	protected float determinePartitionSize() {
		//if any one of the 3 groups has no visible members, then the
		//remaining 2 groups get 180 degrees each.  Otherwise each
		//gets 120 degrees.
		//NOTE: this method probably won't win any awards for
		//elegance, but it gets the job done.
		boolean resourcesAreVisible = false;
		boolean projectsAreVisible = false;
		boolean skillsAreVisible = false;
		float result;
		for (DiffResource p : diffWard.getAllResources()) {
			if (excludeFromRing.contains(p.getResource())) continue;
			resourcesAreVisible = true;
			break;
		}
		for (Skill p : diffWard.getAllSkills()) {
			if (excludeFromRing.contains(p)) continue;
			skillsAreVisible = true;
			break;
		}
		for (DiffProject p : diffWard.getAllProjects()) {
			if (excludeFromRing.contains(p.getProject())) continue;
			projectsAreVisible = true;
			break;
		}
		if (resourcesAreVisible && skillsAreVisible && projectsAreVisible) {
			result = RadialCache.RESOLUTION/3f;
		} else {
			result = RadialCache.RESOLUTION/2f;
		}
		return result;
	}

	/**
	 * In the Diff version, we don't allow the data to be edited,
	 * only viewed. Thus we return false from this method.
	 */
	@Override
	protected boolean checkForOverlappingCards(DartboardCard selectedCard) {
		return false;
	}


}
