package projman;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.geom.Point2D;
import java.io.Serializable;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import projman.DiffWard.Status;

public class FamilyWedge implements Serializable {
	private Entity family;
	private Point2D.Float focalPoint, nameLabelPoint;
	private HyrumPolygon mainPolygon;
	private HyrumPolygon projectInnerPolygon;//shows how much of project is complete
	private HyrumPolygon projectOuterPolygon;//shows 100% of project
	private List<ResourceUtilizationPolygon> resourceUtilizationPolygons;//shows how much time is spent on different projects
	private int[][] gridLinesX;
	private int[][] gridLinesY;
	//private Path2D.Float gridLines[];//TODO replace gridLinesX/Y
	private float start;
	private float length;
	private float idealLength;
	private float idealStart;
	//private boolean dragging;
	private boolean active;
	//private Point2D.Float draggingDelta;
	private String dateText;
	private String progressText;
	private Rectangle progressBox;
	private float progressFraction;
	private int howManyPeopleWithThisSkill=0;
	private static final long serialVersionUID = 42L;
	private Status diffStatus;
	private boolean diffMode;
	private static final Point2D.Float windowCenter = new Point2D.Float();
	private static final DateFormat formatter = new SimpleDateFormat("d MMM yyyy");
	private static final float padding = RadialCache.RESOLUTION/360f;

	public FamilyWedge(Entity f, float start, float length) {
		this(f, start, length, Status.SAME);
		diffMode = false;
	}
	
	public FamilyWedge(Entity f, float start, float length, Status diffStatus) {
		family = f;
		this.start = start;
		this.length = length;
		this.idealLength = length;
		this.idealStart = start;
		active = true;
		this.diffStatus = diffStatus;
		diffMode = true;
		mainPolygon = new HyrumPolygon();
		projectInnerPolygon = new HyrumPolygon();
		projectOuterPolygon = new HyrumPolygon();
		progressBox = new Rectangle();
		gridLinesX = new int[Config.db_numProgressSegments-1][];
		gridLinesY = new int[Config.db_numProgressSegments-1][];
		//gridLines = new Path2D.Float[Config.db_numProgressSegments-1];
		if (f instanceof Project) {
			dateText = formatter.format(((Project)f).getDeadline().getTime());
			progressText = ((Project)f).getPercentComplete() + "%";
			progressFraction = (float)((Project)f).getPercentComplete() / 100f;
		}
		if (f instanceof Skill) {
			howManyPeopleWithThisSkill = ((Skill)f).getMatchingResources().size();
		}
	}
	
	private void buildVertices(RadialCache cache, Dimension windowSize, int offset) {
		int res = RadialCache.RESOLUTION;
		int outermostRadius;
		mainPolygon.reset();
		projectInnerPolygon.reset();
		projectOuterPolygon.reset();
		int mainPolygonRadius1 = 0;//these aren't the REAL radii, just the indices into the cache.
		int mainPolygonRadius2 = 1;
		int projectOuterPolygonRadius1 = mainPolygonRadius2 + 1;
		int projectOuterPolygonRadius2 = projectOuterPolygonRadius1+Config.db_numProgressSegments;
		int projectInnerPolygonRadius1 = projectOuterPolygonRadius1;
		int resourceUtilPolygonRadius1 = projectOuterPolygonRadius1;
		int resourceUtilPolygonRadius2 = resourceUtilPolygonRadius1+2;//TODO make a config value?
		windowCenter.setLocation(windowSize.width/2, windowSize.height/2);

		if (diffMode || !Config.db_frills) {
			outermostRadius = mainPolygonRadius2;
		} else {
			if (family instanceof Project) {
				outermostRadius = projectOuterPolygonRadius2;
			} else if (family instanceof Resource) {
				outermostRadius = resourceUtilPolygonRadius2;
			} else {	//Skill
				outermostRadius = mainPolygonRadius2;
			}
		}
		int index0 = Math.round(offset + start);
		if (index0 < 0) index0 += res;
		int llength = Math.round(length-padding);
		
		//Determine the outer radius of the inner progress polygon.
		//In contrast to similarly-named variables, this one is REAL radius,
		//not an index into an array of radii.
		Point2D.Float p1 = cache.getVertex(projectOuterPolygonRadius1, 0);
		Point2D.Float p2 = cache.getVertex(projectOuterPolygonRadius2, 0);
		Point2D p3 = new Point2D.Float(p1.x+(p2.x-p1.x)*progressFraction,
						p1.y+(p2.y-p1.y)*progressFraction);
		int projectInnerPolygonRadius2 = (int)windowCenter.distance(p3);	
		
		//build main polygon and progress polygon.
		//(Although not terribly efficient, it's easier just to build
		//all the geometry we MIGHT need, rather than check to see if
		//we DO need it.)
		for (int i=0; i<gridLinesX.length; ++i) {
			gridLinesX[i] = new int[llength+1];
			gridLinesY[i] = new int[llength+1];
			//gridLines[i] = new Path2D.Float(Path2D.WIND_EVEN_ODD, llength);
		}

		for (int i=0; i<=llength; ++i) {	
			mainPolygon.addPoint(cache.getVertex(mainPolygonRadius1, (index0 + i) % res).x,
					cache.getVertex(mainPolygonRadius1, (index0 + i) % res).y);
			projectInnerPolygon.addPoint(cache.getVertex(projectInnerPolygonRadius1, (index0 + i) % res).x,
					cache.getVertex(projectInnerPolygonRadius1, (index0 + i) % res).y);
			projectOuterPolygon.addPoint(cache.getVertex(projectOuterPolygonRadius1, (index0 + i) % res).x,
					cache.getVertex(projectOuterPolygonRadius1, (index0 + i) % res).y);
			for (int j=0; j<gridLinesX.length; ++j) {
				gridLinesX[j][i] = (int)(cache.getVertex(projectOuterPolygonRadius1+1+j, (index0 + i) % res).x);
				gridLinesY[j][i] = (int)(cache.getVertex(projectOuterPolygonRadius1+1+j, (index0 + i) % res).y);
			}
		}

		for (int i=llength; i>=0; --i) {
			mainPolygon.addPoint(cache.getVertex(mainPolygonRadius2, (index0 + i) % res).x,
					cache.getVertex(mainPolygonRadius2, (index0 + i) % res).y);
			projectInnerPolygon.addPoint(RadialCache.getSpecialVertex(projectInnerPolygonRadius2, (index0 + i) % res, windowCenter));
			projectOuterPolygon.addPoint(cache.getVertex(projectOuterPolygonRadius2, (index0 + i) % res).x,
					cache.getVertex(projectOuterPolygonRadius2, (index0 + i) % res).y);
		}
		mainPolygon.closePath();
		projectInnerPolygon.closePath();
		projectOuterPolygon.closePath();
		
		//make resource utilization wedges
		if (family instanceof Resource) {
			resourceUtilizationPolygons = new ArrayList<ResourceUtilizationPolygon>();
			Resource person = (Resource)family;
			int proportionalLength = (int)(llength * person.getCapacity());
			List<Team> teams = person.getCompanionships();
			for (Team comp : teams) {
				if (comp.getProject() == null) continue;
				ResourceUtilizationPolygon p = new ResourceUtilizationPolygon(person);
				p.setProject(comp.getProject());
				p.setPercentDevotedToThisProject(person.getPercentageOfTimeSpentOnThisProject(comp));
				//Main.say(family.getName() + " spends " + p.getPercentDevotedToThisProject() + " of the time on " + comp.getProject().getName());
				resourceUtilizationPolygons.add(p);
			}
			//if this resource is not assigned to any projects,
			//create an empty wedge for it just for aesthetics.
			if (resourceUtilizationPolygons.size() == 0) {
				resourceUtilizationPolygons.add(new ResourceUtilizationPolygon(person));
			}
			Collections.sort(resourceUtilizationPolygons, Collections.reverseOrder());
			int counter = 0;
			for (ResourceUtilizationPolygon p : resourceUtilizationPolygons) {
				int len = (int)(proportionalLength*p.getPercentDevotedToThisProject());
				for (int i=counter; i<=counter+len; ++i) {
					p.addPoint(cache.getVertex(resourceUtilPolygonRadius1, (index0 + i) % res).x,
							cache.getVertex(resourceUtilPolygonRadius1, (index0 + i) % res).y);					
				}
				for (int i=counter+len; i>=counter; --i) {
					p.addPoint(cache.getVertex(resourceUtilPolygonRadius2, (index0 + i) % res).x,
							cache.getVertex(resourceUtilPolygonRadius2, (index0 + i) % res).y);
				}
				p.setBaseIndex(counter);
				p.setLength(len);
				p.closePath();
				counter += len;				
			}
		}
		
		//calculate the position of the name label
		focalPoint = cache.getVertex(mainPolygonRadius1, (index0+llength/2) % res);
		Point2D.Float tmp = cache.getVertex(outermostRadius, (index0+llength/2) % res);
		double hypotenuse = tmp.distance(focalPoint);
		double longerHypotenuse = hypotenuse + Config.db_distanceBetweenOuterRadiusAndLabel;
		double ratio = longerHypotenuse / hypotenuse;
		nameLabelPoint = new Point2D.Float(focalPoint.x + (int)(ratio * (tmp.x - focalPoint.x)),
							 focalPoint.y + (int)(ratio * (tmp.y - focalPoint.y)));
		
		//calculate the position of the percent-complete label
		java.awt.geom.Point2D.Float corner1 = cache.getVertex(projectOuterPolygonRadius1, (index0+llength/2) % res);
		java.awt.geom.Point2D.Float corner2 = cache.getVertex(projectOuterPolygonRadius2, (index0+llength/2) % res);
		progressBox.setFrameFromDiagonal(corner1, corner2);
	}

	public synchronized void buildVertices(RadialCache cache, Dimension windowSize, int offset, float swell, float innerRadius) {
		//call the more efficient version of this method if possible.
		if (swell <= 0.001f) {
			buildVertices(cache, windowSize, offset);
			return;
		}
		int res = RadialCache.RESOLUTION;
		float outermostRadius;
		mainPolygon.reset();
		projectInnerPolygon.reset();
		projectOuterPolygon.reset();
		float progressSegmentThickness = Config.db_progressWedgeThickness/Config.db_numProgressSegments;
		float mainPolygonRadius1 = innerRadius;
		float mainPolygonRadius2 = mainPolygonRadius1 + (Config.db_regularWedgeThickness*swell);
		float projectOuterPolygonRadius1 = mainPolygonRadius2 + Config.db_distanceBetweenRegularAndProgressWedge;
		float projectOuterPolygonRadius2 = projectOuterPolygonRadius1+Config.db_progressWedgeThickness;
		float projectInnerPolygonRadius1 = projectOuterPolygonRadius1;
		float resourceUtilPolygonRadius1 = projectOuterPolygonRadius1;
		float resourceUtilPolygonRadius2 = resourceUtilPolygonRadius1+2*progressSegmentThickness;
		windowCenter.setLocation(windowSize.width/2, windowSize.height/2);

		if (diffMode || !Config.db_frills) {
			outermostRadius = mainPolygonRadius2;
		} else {
			if (family instanceof Project) {
				outermostRadius = projectOuterPolygonRadius2;
			} else if (family instanceof Resource) {
				outermostRadius = resourceUtilPolygonRadius2;
			} else {	//Skill
				outermostRadius = mainPolygonRadius2;
			}
		}
		int index0 = Math.round(offset + start);
		if (index0 < 0) index0 += res;
		int llength = Math.round(length-padding);
		
		//Determine the outer radius of the inner progress polygon.
		float projectInnerPolygonRadius2 = projectOuterPolygonRadius1 +
			(projectOuterPolygonRadius2-projectOuterPolygonRadius1)*progressFraction;
		
		//build main polygon and progress polygon.
		//(Although not terribly efficient, it's easier just to build
		//all the geometry we MIGHT need, rather than check to see if
		//we DO need it.)
		for (int i=0; i<gridLinesX.length; ++i) {
			gridLinesX[i] = new int[llength+1];
			gridLinesY[i] = new int[llength+1];
			//gridLines[i] = new Path2D.Float(Path2D.WIND_EVEN_ODD, llength);
		}

		Point2D.Float tmp;
		for (int i=0; i<=llength; ++i) {	
			mainPolygon.addPoint(RadialCache.getSpecialVertex(mainPolygonRadius1, (index0 + i) % res, windowCenter));
			projectOuterPolygon.addPoint(RadialCache.getSpecialVertex(projectOuterPolygonRadius1, (index0 + i) % res, windowCenter));
			projectInnerPolygon.addPoint(RadialCache.getSpecialVertex(projectInnerPolygonRadius1, (index0 + i) % res, windowCenter));
			for (int j=0; j<gridLinesX.length; ++j) {
				tmp = RadialCache.getSpecialVertex(projectOuterPolygonRadius1+
						(1+j)*progressSegmentThickness, (index0 + i) % res, windowCenter);
				gridLinesX[j][i] = (int)tmp.x;
				gridLinesY[j][i] = (int)tmp.y;
			}
		}
		for (int i=llength; i>=0; --i) {
			mainPolygon.addPoint(RadialCache.getSpecialVertex(mainPolygonRadius2, (index0 + i) % res, windowCenter));
			projectOuterPolygon.addPoint(RadialCache.getSpecialVertex(projectOuterPolygonRadius2, (index0 + i) % res, windowCenter));
			projectInnerPolygon.addPoint(RadialCache.getSpecialVertex(projectInnerPolygonRadius2, (index0 + i) % res, windowCenter));
		}
		mainPolygon.closePath();
		projectInnerPolygon.closePath();
		projectOuterPolygon.closePath();
		
		//make resource utilization wedges
		if (family instanceof Resource) {
			resourceUtilizationPolygons = new ArrayList<ResourceUtilizationPolygon>();
			Resource person = (Resource)family;
			int proportionalLength = (int)(llength * person.getCapacity());
			List<Team> teams = person.getCompanionships();
			for (Team comp : teams) {
				if (comp.getProject() == null) continue;
				ResourceUtilizationPolygon p = new ResourceUtilizationPolygon(person);
				p.setProject(comp.getProject());
				p.setPercentDevotedToThisProject(person.getPercentageOfTimeSpentOnThisProject(comp));
				resourceUtilizationPolygons.add(p);
			}
			//if this resource is not assigned to any projects,
			//create an empty wedge for it just for aesthetics.
			if (resourceUtilizationPolygons.size() == 0) {
				resourceUtilizationPolygons.add(new ResourceUtilizationPolygon(person));
			}
			Collections.sort(resourceUtilizationPolygons, Collections.reverseOrder());
			int counter = 0;
			for (ResourceUtilizationPolygon p : resourceUtilizationPolygons) {
				int len = (int)(proportionalLength*p.getPercentDevotedToThisProject());
				for (int i=counter; i<=counter+len; ++i) {
					p.addPoint(RadialCache.getSpecialVertex(resourceUtilPolygonRadius1, (index0 + i) % res, windowCenter));
				}
				for (int i=counter+len; i>=counter; --i) {
					p.addPoint(RadialCache.getSpecialVertex(resourceUtilPolygonRadius2, (index0 + i) % res, windowCenter));
				}
				p.setBaseIndex(counter);
				p.setLength(len);
				p.closePath();
				counter += len;				
			}
		}
		
		//calculate the position of the name label
		focalPoint = RadialCache.getSpecialVertex(mainPolygonRadius1, (index0+llength/2) % res, windowCenter);
		tmp = RadialCache.getSpecialVertex(outermostRadius, (index0+llength/2) % res, windowCenter);
		double hypotenuse = tmp.distance(focalPoint);
		double longerHypotenuse = hypotenuse + Config.db_distanceBetweenOuterRadiusAndLabel;
		double ratio = longerHypotenuse / hypotenuse;
		nameLabelPoint = new Point2D.Float(focalPoint.x + (int)(ratio * (tmp.x - focalPoint.x)),
							 focalPoint.y + (int)(ratio * (tmp.y - focalPoint.y)));
		
		//calculate the position of the percent-complete label
		java.awt.geom.Point2D.Float corner1 = RadialCache.getSpecialVertex(projectOuterPolygonRadius1,
				(index0+llength/2) % res, windowCenter);
		java.awt.geom.Point2D.Float corner2 = RadialCache.getSpecialVertex(projectOuterPolygonRadius2,
				(index0+llength/2) % res, windowCenter);
		progressBox.setFrameFromDiagonal(corner1, corner2);
	}

	public void draw(Graphics2D g) {
		//FIXME the fact that I have to check for this condition at all
		//suggests a threading problem. However, is it worth fixing for
		//a research prototype such as this?  Arguably not.
		if (nameLabelPoint == null) {
			return;
		}
		
		Point2D.Float whereToDrawNameLabel = new Point2D.Float(
				nameLabelPoint.x, nameLabelPoint.y);
		//draw colored, filled wedge
		g.setColor(Config.getColorForEntity(family, diffStatus));
		mainPolygon.fill(g);
		
		if (!diffMode && Config.db_frills) {
			//if (family instanceof Skill) {
			//	//draw number of people with this skill
			//	g.setFont(Config.regularFont);
			//	g.setColor(Config.textColor);
			//	String foo = Integer.toString(howManyPeopleWithThisSkill);
			//	Point2D.Float tangent = new Point2D.Float(nameLabelPoint.x - focalPoint.x, nameLabelPoint.y - focalPoint.y);
			//	double ratio = (0.5 * Config.db_regularWedgeThickness) / (Config.db_regularWedgeThickness + Config.db_distanceBetweenRegularAndProgressWedge);
			//	Point2D.Float textPos = new Point2D.Float((int)(focalPoint.x + ratio*tangent.x), (int)(focalPoint.y + ratio*tangent.y));
			//	g.drawString(foo, textPos.x-g.getFontMetrics().stringWidth(foo)/2,
			//			textPos.y+g.getFontMetrics().getAscent()/2);
			//}
			
			if (family instanceof Resource) {
				//show how much time the person is spending on each project
				for (ResourceUtilizationPolygon p : resourceUtilizationPolygons) {
					//draw empty wedge if resource has no assigned projects
					if (p.getProject() == null) {
						g.setColor(Color.BLACK);
						Stroke prev = g.getStroke();
						g.setStroke(Config.thinLine);
						p.draw(g);
						g.setStroke(prev);
					} else {
						g.setColor(Config.getColorForEntity(p.getProject()));
						p.fill(g);
					}
				}
			}
			
			if (family instanceof Project) {
				g.setColor(Config.getProgressColor((Project)family));
				projectOuterPolygon.fill(g);
				g.setColor(Config.getColorForEntity(family));
				projectInnerPolygon.fill(g);
				g.setColor(Config.backgroundColor);
				for (int i=0; i<gridLinesX.length; ++i) {
					g.drawPolyline(gridLinesX[i], gridLinesY[i], gridLinesX[i].length);
				}
				
				//draw dark outline around wedge
				g.setColor(Config.getBorderColorForEntity(family));
				projectOuterPolygon.draw(g);
				
				//draw "percent complete" text
				g.setFont(Config.largeFont);
				g.setColor(Config.textColor);
				Point center = new Point(progressBox.x+progressBox.width/2, progressBox.y+progressBox.height/2);
				g.drawString(progressText, center.x-g.getFontMetrics().stringWidth(progressText)/2,
										center.y+g.getFontMetrics().getAscent()/2);
				
				//draw deadline text
				g.setFont(Config.regularFont);
				Point2D.Float tangent = new Point2D.Float(nameLabelPoint.x - focalPoint.x, nameLabelPoint.y - focalPoint.y);
				double angle = Math.PI - Math.atan2(tangent.x, tangent.y);
				double ratio = (0.5 * Config.db_regularWedgeThickness) / (Config.db_regularWedgeThickness + Config.db_progressWedgeThickness + 	Config.db_distanceBetweenRegularAndProgressWedge + Config.db_distanceBetweenOuterRadiusAndLabel);
				Point datePoint = new Point((int)(focalPoint.x + ratio*tangent.x), (int)(focalPoint.y + ratio*tangent.y));
				// fudge the angles so that the text always appears rightside up
				if (datePoint.y > windowCenter.y) {
					angle += Math.PI;
				}
				
				g.rotate(angle, datePoint.x, datePoint.y);
				g.drawString(dateText, datePoint.x-g.getFontMetrics().stringWidth(dateText)/2,
						datePoint.y+g.getFontMetrics().getAscent()/2);
				g.rotate(-angle, datePoint.x, datePoint.y);
			}
		}
		
		Stroke prev = g.getStroke();
		if (diffMode) {
			if (diffStatus == Status.NEW) {
				//Stroke prev = g.getStroke();
				g.setStroke(Config.thickLine);
				g.setColor(Config.newEntityColor);
				mainPolygon.draw(g);
				//g.setStroke(prev);
			}
			if (diffStatus != Status.SAME) {
				g.setStroke(Config.thinLine);
			}
		}
		if (active) {
				//draw dark outline around wedge
				g.setColor(Config.getBorderColorForEntity(family));
				mainPolygon.draw(g);
		}
		if (diffMode) {
			g.setStroke(prev);
		}

		g.setFont(Config.regularFont);
		g.setColor(Config.textColor);
		if  (nameLabelPoint.x < windowCenter.x) {
			whereToDrawNameLabel.x -= g.getFontMetrics().stringWidth(family.getName());
		}
		if (nameLabelPoint.y > windowCenter.y) {
			whereToDrawNameLabel.y += g.getFontMetrics().getHeight();
		}
		g.drawString(family.getName(), whereToDrawNameLabel.x, whereToDrawNameLabel.y);
	}
	
	public boolean containsPoint(Point p) {
		if (!active) return false;
		if (mainPolygon.contains(p)) return true;
		if (family instanceof Project) {
			return (projectOuterPolygon.contains(p));
		}
		return false;
	}
	
//	public void drag(Point delta) {
//		draggingDelta.setLocation(draggingDelta.x+delta.x,
//				draggingDelta.y+delta.y);
//	}
	
//	public void setSelected(boolean selected) {
//		dragging = selected;
//		draggingDelta.setLocation(0,0);
//	}
	
	public Entity getEntity() {
		return family;
	}
	
	@Override
	public String toString() {
		return family.getName();
	}
	
	public void setActive(boolean a) {
		active = a;
	}

	public Point2D.Float getFocalPoint() {
		return focalPoint;
	}
	
	public Point2D.Float getPointAlongInnerRadius(int proportionalIndex, int divisions) {
		int numPoints = mainPolygon.getNumPoints()/2;
		int divisionLength = numPoints / divisions;
		int index = divisionLength*proportionalIndex + (divisionLength/2);
		return mainPolygon.getPointAt(index);//TODO maybe change to a copy?
	}
	
	public Point2D.Float getPointClosestToAssignedProject(Project project) {
		Point2D.Float result = null;
		for (ResourceUtilizationPolygon p : resourceUtilizationPolygons) {
			if (p.getProject().equals(project)) {
				int index = p.getMiddleIndex();
				result = mainPolygon.getPointAt(index);
				break;
			}
		}
		return result;
	}
	
	public String getTooltipText(Point pos) {
		String result = null;
		if (!diffMode) {
			if (family instanceof Resource) {
				if (mainPolygon.contains(pos)) {
					int count = resourceUtilizationPolygons.size();
					if (count == 1) {
						Project p = resourceUtilizationPolygons.get(0).getProject();
						if (p == null) {
							result = family.getName() + " is not currently assigned to any projects.";
						} else {
							result = family.getName() + " is currently assigned to one project, " + p.getName() + ".";
						}
					} else {
						result = family.getName() + " is currently assigned to " + count + " projects.";
					}
				} else {
					for (ResourceUtilizationPolygon p : resourceUtilizationPolygons) {
						if (p.contains(pos)) {
							if (p.getProject() != null) {
								result = p.getResource() + " works on " + p.getProject().getName()
										+ " " + Math.round(p.getPercentDevotedToThisProject()*100)
										+ "% of the time.";
							} else {
								result = p.getResource() + " is not currently assigned to any projects.";
							}
							break;
						}
					}
				}
			} else if (family instanceof Skill) {
				if (mainPolygon.contains(pos)) {
					result = "You have ";
					switch (howManyPeopleWithThisSkill) {
					case 0:
						result += "no workers";
						break;
					case 1:
						result += "one worker";
						break;
					default:
						result += (howManyPeopleWithThisSkill + " workers");
					}
					result += " qualified as \"" + family.getName() + "\".";
				}
			} else {	//Project
				if (mainPolygon.contains(pos) || projectOuterPolygon.contains(pos)) {
					result = family.getName() + "'s due date is " + dateText + ".";
				}
			}
		}
		return result;
	}
	
	float getLength() {
		return length;
	}
	
	float getStart() {
		return start;
	}

	void resetStart(float s) {
		start = s;
	}

	void adjustStartByDelta(float delta) {
		start += delta;
	}
	
	void adjustLengthByDelta(float delta) {
		length += delta;
	}

	void resetLength(float len) {
		this.length = len;
	}
	
	void reset() {
		length = idealLength;
		start = idealStart;
	}
	
	float getIdealLength() {
		return idealLength;
	}
	
	public Status getDiffStatus() {
		return diffStatus;
	}
	
	public Point2D.Float getStartPoint() {
		return mainPolygon.getPointAt(0);
	}
	
}
