package projman;

import java.awt.Dialog;
import java.awt.Frame;
import java.io.Serializable;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EventListener;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JOptionPane;

public class Ward implements Serializable {
	private List<Project> projects;
	private List<Resource> resources;
	private List<Skill> knownSkills;
	private List<WardListener> listeners;
	private static final long serialVersionUID = 42L;
	
	public Ward() {
		projects = new ArrayList<Project>();
		resources = new ArrayList<Resource>();
		knownSkills = new ArrayList<Skill>();
		listeners = new ArrayList<WardListener>();
	}
	
	public void addProject(Project f) {
		projects.add(f);
		familyChanged();
	}
	
	public void addResource(Resource f) {
		resources.add(f);
		familyChanged();
	}

	public void addSkill(Skill f) {
		knownSkills.add(f);
		familyChanged();
	}

	public boolean containsEntity(Entity tot) {
		return (projects.contains(tot) || 
			   resources.contains(tot) || 
			   knownSkills.contains(tot));
	}
	
	public List<Project> getAllProjects() {
		return projects;
	}
	
	public List<Resource> getAllResources() {
		return resources;
	}
	
	public List<Skill> getAllSkills() {
		return knownSkills;
	}
	
	public boolean addWardListener(WardListener wl) {
		return listeners.add(wl);
	}
	
	public List<WardListener> clearWardListeners() {
		List<WardListener> backup = new ArrayList<WardListener>(listeners);
		listeners.clear();
		//listeners = null;
		return backup;
	}
	
	public void restoreWardListeners(List<WardListener> backup) {
		listeners.clear();
		listeners.addAll(backup);
		//listeners = new ArrayList<WardListener>(backup);
	}
	
	public Project getProjectByName(String name) {
		Project result = null;
		for (Project p : projects) {
			if (p.getName().equals(name)) {
				result = p;
				break;
			}
		}
		return result;
	}
	
	public Resource getResourceByName(String name) {
		Resource result = null;
		for (Resource p : resources) {
			if (p.getName().equals(name)) {
				result = p;
				break;
			}
		}
		return result;
	}
	
	public void familyChanged() {
		Collections.sort(projects);
		Collections.sort(resources);
		Collections.sort(knownSkills);
		fireWardChanged();
	}
	
	public void assignmentsChanged() {
		fireWardChanged();
	}
	
	private void fireWardChanged() {
		for (WardListener el : listeners) {
			el.wardChanged();
		}
	}
	
	public interface WardListener extends EventListener {
		public void wardChanged();
	}

	public void displayNewProjectDialog(Frame parent, Auxiliary currentAuxiliary) {
		displayNewProjectDialog_helper(new EditProjectDialog(parent, false, new Project(this)));
	}

	public void displayNewProjectDialog(Dialog parent, Auxiliary currentAuxiliary) {
		displayNewProjectDialog_helper(new EditProjectDialog(parent, false, new Project(this)));
	}

	private void displayNewProjectDialog_helper(EditProjectDialog fd) {
		Project newFamily = fd.getProject();
		if (newFamily != null && newFamily.isChanged()) {
			//Main.say("project changed!" + newFamily.toString());
			addProject(newFamily);
		}
	}
	
	public void displayEditProjectDialog(Frame parent, Project fam) {
		new EditProjectDialog(parent, true, fam);
		//Main.say("project changed!" + fam.toString());
		familyChanged();
	}
	
	public void displayEditProjectDialog(Dialog parent, Project fam) {
		new EditProjectDialog(parent, true, fam);
		//Main.say("project changed!" + fam.toString());
		familyChanged();
	}
	
	public void displayEditSomethingDialog(Entity obj) {
		if (obj instanceof Project) {
			displayEditProjectDialog(Main.instance(), (Project)obj);
		}
		if (obj instanceof Resource) {
			displayEditResourceDialog(Main.instance(), (Resource)obj);
		}
		if (obj instanceof Skill) {
			displayEditSkillDialog(Main.instance(), (Skill)obj);
		}
	}

	
	public void deleteProjects(DefaultListModel model, int indices[]) {
		if (indices.length == 0) return;
		String confirmationMessage;
		if (indices.length == 1) {
			Object[] messageArguments = {((Project)model.get(indices[0])).getName()};
			MessageFormat formatter = new MessageFormat("");
			formatter.applyPattern(Main.messages.getString(Constants.CONFIRM_DELETE_PERSON));
			confirmationMessage = formatter.format(messageArguments);
		} else {
			confirmationMessage = Main.messages.getString(Constants.CONFIRM_DELETE_PROJECTS);
		}
		int yesno = JOptionPane.showConfirmDialog(Main.instance(), confirmationMessage,
				Main.messages.getString(Constants.DELETE_PROJECT_DIALOG_TITLE),
				JOptionPane.YES_NO_OPTION);
		if (yesno == JOptionPane.YES_OPTION) {
			//TODO maybe give the user the option of disbanding the whole team,
			//or just removing the project and keeping the resources together?
			deleteOneOrMoreEntities(model, indices);
//			int indexToRemove = 0;
//			int count = 0;
//			for (int i : indices) {
//				indexToRemove = i - count;
//				Project condemned = (Project)model.remove(indexToRemove);
//				//projects.remove(condemned);
//				deleteEntity(condemned);
//				++count;
//			}
//			familyChanged();
		}
	}
	
	private void deleteOneOrMoreEntities(DefaultListModel model, int indices[]) {
		int indexToRemove = 0;
		int count = 0;
		for (int i : indices) {
			indexToRemove = i - count;
			Entity condemned = (Entity)model.remove(indexToRemove);
			//projects.remove(condemned);
			deleteEntity(condemned);
			++count;
		}
		familyChanged();
	}
	
//	public int countEntities() {
//		return (projects.size() + resources.size());
//	}
	
	public void displayNewResourceDialog(Frame parent, Auxiliary currentAuxiliary) {
		displayNewResourceDialog_helper(new EditResourceDialog(parent, false, new Resource(this)));
	}

	public void displayNewResourceDialog(Dialog parent, Auxiliary currentAuxiliary) {
		displayNewResourceDialog_helper(new EditResourceDialog(parent, false, new Resource(this)));
	}

	private void displayNewResourceDialog_helper(EditResourceDialog fd) {
		Resource newFamily = fd.getResource();
		if (newFamily != null && newFamily.isChanged()) {
			addResource(newFamily);
		}
	}
	
	public void displayEditResourceDialog(Frame parent, Resource fam) {
		new EditResourceDialog(parent, true, fam);
		familyChanged();
	}
	
	public void displayEditResourceDialog(Dialog parent, Resource fam) {
		new EditResourceDialog(parent, true, fam);
		familyChanged();
	}
	
	public void deleteResources(DefaultListModel model, int indices[]) {
		if (indices.length == 0) return;
		String confirmationMessage;
		if (indices.length == 1) {
			Object[] messageArguments = {((Resource)model.get(indices[0])).getName()};
			MessageFormat formatter = new MessageFormat("");
			formatter.applyPattern(Main.messages.getString(Constants.CONFIRM_DELETE_PERSON));
			confirmationMessage = formatter.format(messageArguments);
		} else {
			confirmationMessage = Main.messages.getString(Constants.CONFIRM_DELETE_RESOURCES);
		}
		int yesno = JOptionPane.showConfirmDialog(Main.instance(), confirmationMessage,
				Main.messages.getString(Constants.DELETE_RESOURCE_DIALOG_TITLE),
				JOptionPane.YES_NO_OPTION);
		if (yesno == JOptionPane.YES_OPTION) {
			deleteOneOrMoreEntities(model, indices);
//			int indexToRemove = 0;
//			int count = 0;
//			for (int i : indices) {
//				indexToRemove = i - count;
//				Resource condemned = (Resource)model.remove(indexToRemove);
//				//resources.remove(condemned);
//				deleteEntity(condemned);
//				++count;
//			}
//			familyChanged();
		}
	}

	public void displayNewSkillDialog(Frame parent) {
		displayNewSkillDialog_helper(new EditSkillDialog(parent, false, new Skill(this)));
	}

	public void displayNewSkillDialog(Dialog parent) {
		displayNewSkillDialog_helper(new EditSkillDialog(parent, false, new Skill(this)));
	}

	private void displayNewSkillDialog_helper(EditSkillDialog fd) {
		Skill newFamily = fd.getSkill();
		if (newFamily != null && newFamily.isChanged()) {
			addSkill(newFamily);
		}
	}
	
	public void displayEditSkillDialog(Frame parent, Skill fam) {
		new EditSkillDialog(parent, true, fam);
		familyChanged();
	}
	
	public void displayEditSkillDialog(Dialog parent, Skill fam) {
		new EditSkillDialog(parent, true, fam);
		familyChanged();
	}
	
	public void deleteSkills(DefaultListModel model, int indices[]) {
		if (indices.length == 0) return;
		String confirmationMessage;
		if (indices.length == 1) {
			Object[] messageArguments = {((Skill)model.get(indices[0])).getName()};
			MessageFormat formatter = new MessageFormat("");
			formatter.applyPattern(Main.messages.getString(Constants.CONFIRM_DELETE_SKILL));
			confirmationMessage = formatter.format(messageArguments);
		} else {
			confirmationMessage = Main.messages.getString(Constants.CONFIRM_DELETE_SKILLS);
		}
		int yesno = JOptionPane.showConfirmDialog(Main.instance(), confirmationMessage,
				Main.messages.getString(Constants.DELETE_SKILL_DIALOG_TITLE),
				JOptionPane.YES_NO_OPTION);
		if (yesno == JOptionPane.YES_OPTION) {
			deleteOneOrMoreEntities(model, indices);
//			int indexToRemove = 0;
//			int count = 0;
//			for (int i : indices) {
//				indexToRemove = i - count;
//				Skill condemned = (Skill)model.remove(indexToRemove);
//				//knownSkills.remove(condemned);
//				deleteEntity(condemned);
//				++count;
//			}
//			familyChanged();
		}
	}
	
	private void deleteEntity(Entity condemned) {
		if (condemned instanceof Project) {
			Project cp = (Project)condemned;
			Team team = cp.getTeam();
			//TODO we might not want to disband the team
			//in the HT version of this program?
			if (team != null) {
				team.disband();
			}
			projects.remove(condemned);
		}
		if (condemned instanceof Resource) {
			Resource cr = (Resource)condemned;
			Team[] projectTeams = cr.getCompanionshipsAsArray();
			for (Team team : projectTeams) {
				//Main.say("Removing " + cr.getName() + " from team " + team.getProject().getName());
				team.removeResource(cr);
			}
			resources.remove(condemned);
		}
		if (condemned instanceof Skill) {
			Skill cs = (Skill)condemned;
			for (Project p : projects) {
				p.removeAllSkillRequirementsFor(cs);
			}
			for (Resource r : resources) {
				r.removeSkill(cs);
			}
			knownSkills.remove(condemned);
		}
	}

}