package projman;


import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import projman.AbstractWhiteboard.ChartStyle;

public class WhiteboardTeam implements Serializable {
	private Rectangle bounds;
	private Team team;
	private AbstractWhiteboard parent;
	private List<WhiteboardCard> cards;
	private List<WhiteboardCard> teacherCards;
	private List<WhiteboardCard> teacheeCards;
	private final int MAX_CARDS_PER_LINE = 3;
	private static int VOID = Integer.MAX_VALUE;
	private int teacherLinePos = VOID;
	private int teacheeLinePos = VOID;
	private Point centerPoint;
	private static final long serialVersionUID = 42L;

	public WhiteboardTeam(AbstractWhiteboard parent, Point p) {
		this.team = new Team();
		bounds = new Rectangle(p);
		cards = new ArrayList<WhiteboardCard>();
		teacherCards = new ArrayList<WhiteboardCard>();
		teacheeCards = new ArrayList<WhiteboardCard>();
		this.parent = parent;
		centerPoint = null;
	}

	public boolean canAccept(WhiteboardCard card) {
		boolean result = false;

		//Is this card close to this team?  Good!
		if (bounds.intersects(card.getBounds())) result = true;

		//Does this team already contain this card? Bad!
		if (team.contains(card.getEntity())) return false;
		
		//If a project card, does this team already have
		//a project? (PM only; does not apply to HT version.)
		if ((card.getEntity() instanceof Project) &&
				(!(teacherCards.isEmpty()))) return false;

		return result;
	}

	public void add(WhiteboardCard card) {
		cards.add(card);
		//We switch the teacher/teachee card pile in this version
		//of the program to make the charts look better
		if (card.getTeachingStatus() == EntityStatus.PROJECT) {
			teacheeCards.add(card);
		} else {
			teacherCards.add(card);
		}
		card.assignToTeam(this);
		recalibrate();
	}
	
	/**
	 * Used for adding automatically generated cards to a team.
	 * For example, if a resource is added to a project in the 
	 * dartboard view, we need to create a card and put it in
	 * the whiteboard view.
	 * @param card the card to add
	 */
	public void softAdd(WhiteboardCard card) {
		cards.add(card);
		if (card.getTeachingStatus() == EntityStatus.PROJECT) {
			teacheeCards.add(card);
		} else {
			teacherCards.add(card);
		}
		card.setAffiliatedTeam(this);
		recalibrate();
	}


	public void remove(WhiteboardCard card) {
		cards.remove(card);
		if (card.getTeachingStatus() == EntityStatus.PROJECT) {
			teacheeCards.remove(card);
		} else {
			teacherCards.remove(card);
		}
		card.removeFromTeam(this);
		recalibrate();
	}
	
	/**
	 * Used for removing cards from a team when the corresponding
	 * entity is removed from the team via the dartboard.
	 * @param card the card to remove
	 */
	private void softRemove(WhiteboardCard card) {
		cards.remove(card);
		if (card.getTeachingStatus() == EntityStatus.PROJECT) {
			teacheeCards.remove(card);
		} else {
			teacherCards.remove(card);
		}
		//card.removeFromTeam(this);
		recalibrate();
	}



	public boolean contains(WhiteboardCard card) {
		return (cards.contains(card));
	}

	public WhiteboardCard getCardAtPosition(Point pos) {
		for (WhiteboardCard c : cards) {
			if (c.containsPoint(pos)) {
				return c;
			}
		}
		return null;
	}
	
	public WhiteboardCard getMatchingCard(Entity entity, EntityStatus ts) {
		WhiteboardCard card = null;
		//Keeping the TeachingStatus parameter around for future use;
		//we don't need it right now. (Maybe in HT program?)
		for (WhiteboardCard wbc : teacheeCards) {
			if (wbc.getEntity() == entity) {
				card = wbc;
				break;
			}
		}
		return card;
	}
	
	public void setTopLeftCorner(final int cornerX, final int cornerY) {
		int newCenterX = cornerX + bounds.width/2;
		int newCenterY = cornerY;// + bounds.height/2;
		Point delta = new Point(newCenterX - centerPoint.x, newCenterY - centerPoint.y);
		translate(delta);
	}
	
	public void translate(Point delta) {
		centerPoint.translate(delta.x, delta.y);
		teacherLinePos += delta.y;
		teacheeLinePos += delta.y;
		recalibrate();
	}

	public void recalibrate() {
		int buffer = Config.wb_distanceBetweenCards;
		int smallBuffer = buffer - buffer/3;
		bounds.setBounds(0,0,0,0);	//reset bounds

		switch (parent.getChartStyle()) {
		case HORIZONTAL:
			break;

		case VERTICAL:
		default:

			if (centerPoint == null) {
				WhiteboardCard anchor = cards.get(0);
				bounds.setBounds(anchor.getBounds());
				centerPoint = new Point((int)bounds.getCenterX(), (int)bounds.getCenterY());
				if (teacheeCards.contains(anchor)) {
					teacheeLinePos = bounds.y;
					teacherLinePos = teacheeLinePos - smallBuffer;
				} else {
					teacherLinePos = bounds.y+bounds.height;
					teacheeLinePos = teacherLinePos + smallBuffer;
				}
				return;
			}
		//hack
		if (teacherCards.size() <= 1 || teacheeCards.size() <= 1 ) {
			teacherLinePos = teacheeLinePos - smallBuffer;
		} else {
			teacherLinePos = teacheeLinePos - buffer;
		}

		int teacherRowWidth = 0;
		int teacheeRowWidth = 0;
		for (WhiteboardCard c : teacherCards) {
			teacherRowWidth += c.getBounds().width;
		}

		int currentTeacheeRowWidth = 0;
		int i=0;
		while (i<teacheeCards.size()) {
			for (int j=0; j<MAX_CARDS_PER_LINE && i<teacheeCards.size(); ++j, ++i) {
				currentTeacheeRowWidth += teacheeCards.get(i).getBounds().width;
			}
			if (currentTeacheeRowWidth > teacheeRowWidth) {
				teacheeRowWidth = currentTeacheeRowWidth;
			}
			currentTeacheeRowWidth = 0;
		}
		teacherRowWidth += (smallBuffer * Math.max(0, teacherCards.size()-1));
		teacheeRowWidth += (smallBuffer * Math.min(MAX_CARDS_PER_LINE-1, Math.max(0, teacheeCards.size()-1)));
		bounds.width = Math.max(teacherRowWidth, teacheeRowWidth);

		Point teacherStartingPos = new Point(centerPoint.x-teacherRowWidth/2, teacherLinePos);
		Point teacheeStartingPos = new Point(centerPoint.x-teacheeRowWidth/2, teacheeLinePos);
		bounds.x = Math.min(teacherStartingPos.x, teacheeStartingPos.x);

		//NOW place the cards where they belong
		Point currentPos = new Point(teacherStartingPos);
		for (WhiteboardCard c : teacherCards) {//project
			c.setBottomLeftPosition(currentPos);
			currentPos.translate(c.getBounds().width+smallBuffer, 0);
			bounds.height = Math.max(bounds.height, c.getBounds().height);
		}
		currentPos.setLocation(teacheeStartingPos);
		int maxTeacheeCardHeightInThisRow = 0;
		i=0;
		for (WhiteboardCard c : teacheeCards) {//resources
			c.setTopLeftPosition(currentPos);
			currentPos.translate(c.getBounds().width+smallBuffer, 0);
			maxTeacheeCardHeightInThisRow = Math.max(maxTeacheeCardHeightInThisRow, c.getBounds().height + smallBuffer);
			++i;
			if (i % MAX_CARDS_PER_LINE == 0) {
				currentPos.x = teacheeStartingPos.x;
				currentPos.translate(0, maxTeacheeCardHeightInThisRow/* + smallBuffer*/);
				bounds.height += maxTeacheeCardHeightInThisRow;
				maxTeacheeCardHeightInThisRow = 0;
			}
		}
		bounds.height += maxTeacheeCardHeightInThisRow;

		//TODO change this in HT version of program
		if (teacherCards.size() > 0) {
			bounds.y = teacherCards.get(0).getBounds().y;
		} else if (teacheeCards.size() > 0) {
			bounds.y = teacheeCards.get(0).getBounds().y;
		}

		break;
		}

	}

	public void draw(Graphics2D g) {
		ChartStyle style = parent.getChartStyle();
		Point start, end;
		for (int i=0; i<teacherCards.size(); ++i) {
			WhiteboardCard card = teacherCards.get(i);
			//TODO this is probably not very efficient.
			//We can probably get away with recalibrating only
			//once; after all the cards have been "prepared".
			//Make it faster, if time ever becomes an issue.
			if (card.isChanged()) {
				card.prepareToDraw(g);
				recalibrate();
			}
			card.draw(g);

			//draw a line between teachers if there are no teachees,
			//just so the user gets an immediate sense of connectivity
			if (i<teacherCards.size()-1 && teacheeCards.size() == 0) {
				start = card.getTeacherConnectionPoint1(style);
				end = teacherCards.get(i+1).getTeacherConnectionPoint2(style);
				g.setColor(Config.wb_teamConnectionColor);
				g.drawLine(start.x, start.y, end.x, end.y);
			}
		}

		for (int i=0; i<teacheeCards.size(); ++i) {
			WhiteboardCard card = teacheeCards.get(i);
			if (card.isChanged()) {
				card.prepareToDraw(g);
				recalibrate();
			}
			card.draw(g);
		}

		if (teacheeCards.size() > 0 && teacherCards.size() > 0) {
			g.setColor(Config.wb_teamConnectionColor);

			//draw lines from the teachers, toward the teachees
			for (WhiteboardCard card : teacherCards) {
				start = card.getTeacherToTeacheeConnectionPoint1(style);
				end = card.getTeacherToTeacheeConnectionPoint2(style);
				g.drawLine(start.x, start.y, end.x, end.y);				
			}

			//draw a line connecting the lines we just drew
			Point teacherLine1 = teacherCards.get(0).getTeacherToTeacheeConnectionPoint2(style);
			Point teacherLine2 = teacherCards.get(teacherCards.size()-1).getTeacherToTeacheeConnectionPoint2(style);
			g.drawLine(teacherLine1.x, teacherLine1.y, teacherLine2.x, teacherLine2.y);
			//draw lines from the teachees, toward the teachers
			int last = Math.min(teacheeCards.size(), MAX_CARDS_PER_LINE)-1;
			for (int j=0; j<=last; ++j) {
				WhiteboardCard card = teacheeCards.get(j);
				start = card.getTeacheeToTeacherConnectionPoint1(style);
				end = card.getTeacheeToTeacherConnectionPoint2(style);
				g.drawLine(start.x, start.y, end.x, end.y);
			}
			//draw a line connecting the lines we just drew
			Point teacheeLine1 = teacheeCards.get(0).getTeacheeToTeacherConnectionPoint2(style);
			Point teacheeLine2 = teacheeCards.get(last).getTeacheeToTeacherConnectionPoint2(style);
			g.drawLine(teacheeLine1.x, teacheeLine1.y, teacheeLine2.x, teacheeLine2.y);

			//now draw a line connecting the teachers to the teachees
			Point mid1 = getMidPoint(teacherLine1, teacherLine2);
			Point mid2 = getMidPoint(teacheeLine1, teacheeLine2);
			//g.drawLine(mid1.x, mid1.y, mid1.x, mid2.y);
			g.drawLine(mid1.x, mid1.y, mid2.x, mid2.y);

			for (int i=last+1; i<teacheeCards.size(); ++i) {
				start = teacheeCards.get(i-MAX_CARDS_PER_LINE).getTeacheeToTeacheeConnectionPoint1(style);
				end = teacheeCards.get(i).getTeacheeToTeacheeConnectionPoint2(style);
				g.drawLine(start.x, start.y, end.x, end.y);
			}
		}


//		if (selected) {
//		g.setColor(Config.wb_selectedTeamBorderColor);
//		g.drawRect(bounds.x, bounds.y, bounds.width, bounds.height);
//		}
	}

	public Team getTeam() {
		return team;
	}
	
	public void setCompanionshipManually(Team comp) {
		team = comp;
	}

	public boolean isEmpty() {
		return (cards.isEmpty());
	}

	public void compareWithWard(Ward ward, Graphics2D g) {
		List<WhiteboardCard> condemned = new ArrayList<WhiteboardCard>();
		Map<Resource, WhiteboardCard> accountedFor = new HashMap<Resource, WhiteboardCard>();
		for (WhiteboardCard card : cards) {
			Entity entity = card.getEntity();
			//check whether the ward contains the family that this
			//card belongs to.  If not, delete the card.
			//Else, check whether the person on the card belongs to
			//the family. If not, delete the card.
			if (!ward.containsEntity(entity)) {
				condemned.add(card);
			} else {
				if (entity.isChanged()) {
					card.recalculateSize(g);
				}
				if (entity instanceof Resource) {
					accountedFor.put((Resource)entity, card);
				}
			}
		}
		//actually do the deleting
		for (WhiteboardCard wc : condemned) {
			remove(wc);
		}
		
		//check to see if anyone's been added to the team.
		//If so, add a new card for them.
		if (accountedFor.size() != team.getResources().size()) {
			List<Resource> teamPlayers = team.getResources();
			if (teamPlayers.size() > accountedFor.size()) {
				for (Resource r : teamPlayers) {
					if (!(accountedFor.keySet().contains(r))) {
						WhiteboardCard card = new WhiteboardCard(r,
								Resource.getTeachingStatus(), centerPoint, g);
						softAdd(card);
					}
				}
			} else {
				for (Resource r : accountedFor.keySet()) {
					if (!(teamPlayers.contains(r))) {
						softRemove(accountedFor.get(r));
					}
				}
			}
		}
		
	}

	public boolean contains(Point p) {
		return bounds.contains(p);
	}

//	public void setSelected(boolean selected) {
//		this.selected = selected;
//	}

	private Point getMidPoint(Point p1, Point p2) {
		return new Point((p1.x+p2.x)/2, (p1.y+p2.y)/2);
	}

	//This method is grossly inefficient.
	//But unless it becomes a bottleneck,
	//does it need to be efficient?  No.
	public static Rectangle computeUnion(List<WhiteboardTeam> list) {
		if (list.isEmpty()) return new Rectangle();
		Rectangle result = new Rectangle(list.get(0).bounds);
		for (WhiteboardTeam wt : list) {
			result = result.union(wt.bounds);
		}
		return result;
	}

	public boolean intersects(Rectangle r) {
		return bounds.intersects(r);
	}
	
	public Dimension getDimensions() {
		return bounds.getSize();
	}
	
	@Override
	public String toString() {
		return team.toString();
	}
	
	public String getProjectName() {
		String result = "";
		WhiteboardCard projectCard = teacherCards.get(0);
		if (projectCard != null) {
			result = projectCard.getName();
		}
		return result;
	}
	
}
