package projman;

import java.awt.Cursor;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

import javax.swing.TransferHandler;

public class WhiteboardWidget extends AbstractWhiteboard {
	
    private static final long serialVersionUID = 42L;
    private Map<Auxiliary, List<WhiteboardTeam>> allTeams;
    private Ward ward;
    private WhiteboardCard selectedCard = null;
    private WhiteboardTeam teamContainingSelectedCard = null;
    private WhiteboardCard ghostCard = null;
	
	public WhiteboardWidget(Ward ward) {
		super();
		allTeams = new HashMap<Auxiliary, List<WhiteboardTeam>>();
		allTeams.put(Auxiliary.ELDERS_QUORUM, new ArrayList<WhiteboardTeam>());
		allTeams.put(Auxiliary.HIGH_PRIESTS_GROUP, new ArrayList<WhiteboardTeam>());
		allTeams.put(Auxiliary.RELIEF_SOCIETY, new ArrayList<WhiteboardTeam>());
		this.ward = ward;
		mouse = new WhiteboardWidgetMouseHandler();
		addMouseMotionListener(mouse);
		addMouseListener(mouse);
	}
	
	public WhiteboardCard getSelectedCard() {
		return selectedCard;
	}
	
	public void handleDrop(Entity entity, EntityStatus status, Point loc) {		
		ghostCard = null;
		WhiteboardCard card = new WhiteboardCard(entity, status, loc, getGraphics2D());
		cardMoved(card);
	}
	
	private void cardMoved(WhiteboardCard card) {
		boolean cardJoinedExistingCompanionship = 
			updateTeamsBasedOnNewCardPosition(card);
		
		if (!cardJoinedExistingCompanionship) {
			//create default Companionships because the card wasn't dropped
			//close to any existing companionships.
			WhiteboardTeam wt = new WhiteboardTeam(this, card.getPosition());
			wt.add(card);
			teams.add(wt);
			//card.setPosition(loc);
		}
		
		//refresh the CandidateList and redraw the chart
		ward.assignmentsChanged();
		repaint();
		
	}
	
	public void restoreCardAfterBadDrop() {
		//TODO maybe add some quick animation
		//effect to show the card flying back
		//from where you dropped it, to where it
		//is going, so that the user keeps his
		//bearings?
//		Main.say("bad drop: putting " + selectedCard.getName() + " back where it was");
		Point prev = selectedCard.getPrevPosition();
		selectedCard.setPosition(prev);
		teamContainingSelectedCard.add(selectedCard);
		cleanupAfterDrop();
		repaint();
	}
	
	public void cleanupAfterDrop() {
		//Main.say("cleanupAfterDrop");
		((WhiteboardWidgetMouseHandler)mouse).dragging = false;
		((WhiteboardWidgetMouseHandler)mouse).dnd = false;
		selectedCard = null;
		ghostCard = null;
		repaint();
	}
	
	
	private class WhiteboardWidgetMouseHandler extends WhiteboardMouseHandler {
		boolean dragging = false;//are we dragging a single card?
		boolean dnd = false;//are we dragging to/from the candidate list?
		boolean didCardMove = false;
		
		@Override
		public void mousePressed(MouseEvent e) {
			mousePos = e.getPoint();
			prevMousePos = mousePos;
			didCardMove = false;
			if (e.getButton() == 1) {
				if (lassoRect.contains(mousePos)) {
					return;
				}
				for (WhiteboardTeam wt : teams) {
					WhiteboardCard c = wt.getCardAtPosition(mousePos);
					if (c != null) {
						select(c);
						dragging = true;
						dnd = false;
						c.setPrevPosition(c.getPosition());
						return;
					}
				}

				//if we've reached this point, then the click was
				//outside any existing team boundaries
				lassoPos = new Point(mousePos);
				lassoRect.setLocation(lassoPos);
				lassoRect.setSize(0,0);//necessary?
				draggedTeams.clear();
				setCursor(Cursor.getDefaultCursor());
			}
			//TODO: put this back in after the InfoVis deadline
//			if (e.getButton() == 3) {
//				for (WhiteboardTeam wt : teams) {
//					WhiteboardCard c = wt.getCardAtPosition(mousePos);
//					if (c != null) {
//						c.drawPopupMenu(WhiteboardWidget.this, e.getPoint());
//					}
//				}
//			}
		}
		
		@Override
		public void mouseReleased(MouseEvent e) {
			mousePos = e.getPoint();
			dnd = false;
//			Main.say("mouse released at " + mousePos);
			setCursor(Cursor.getDefaultCursor());
			if (dragging) {
				dragging = false;
				if (selectedCard != null) {
					//if the user dragged the card outside the window,
					//put it back where it was before the drag started.
					//if the user keeps the card inside the window,
					//remove it from its team, and add it to whatever
					//team it's closest to.
					//If the user dropped the card back in the same
					//team it was already in, then leave it alone.					
					Rectangle bounds = new Rectangle(windowSize);
					if (bounds.contains(mousePos)) {
						if (teamContainingSelectedCard.contains(mousePos)) {
							teamContainingSelectedCard.recalibrate();
							if (!didCardMove) {
								WhiteboardWidget.this.highlightSimilarCards(selectedCard.getEntity(),
										selectedCard.getTeachingStatus());
							}
						} else {
							teamContainingSelectedCard.remove(selectedCard);
							cardMoved(selectedCard);
						}
					} else {
						restoreCardAfterBadDrop();
					}
					selectedCard = null;
				}
				repaint();
			}
			if (!draggedTeams.isEmpty()) {
				lassoRect = new Rectangle();
				draggedTeams.clear();
				repaint();
			}
			if (lassoPos != null) {
				//select all teams that fall within
				//the lasso's boundaries
				for (WhiteboardTeam wt : teams) {
					if (wt.intersects(lassoRect)) {
						draggedTeams.add(wt);
						//wt.setSelected(true);
					}
				}
				if (draggedTeams.isEmpty()) {
					lassoRect = new Rectangle();
				} else {
					setCursor(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
					lassoRect = WhiteboardTeam.computeUnion(draggedTeams);
				}
				lassoPos = null;
				repaint();
			}
		}
		
		@Override
		public void mouseDragged(MouseEvent e) {
			mousePos = e.getPoint();
			Point delta = new Point(mousePos.x-prevMousePos.x,
					mousePos.y-prevMousePos.y);
			if (dragging && !dnd) {
				if (selectedCard != null) {
					selectedCard.drag(delta);
					didCardMove = true;
					if (mousePos.x <= 0) {
						dnd = true;
						teamContainingSelectedCard.remove(selectedCard);
						TransferHandler handler = getTransferHandler();
						//Tell the transfer handler to initiate the drag.
						handler.exportAsDrag(WhiteboardWidget.this, e, TransferHandler.COPY);
					}
				}
			}
			if (!draggedTeams.isEmpty()) {
				for (WhiteboardTeam wt : draggedTeams) {
					wt.translate(delta);
				}
				lassoRect.translate(delta.x, delta.y);
			}
			prevMousePos = mousePos;
			repaint();
		}
	}
	
	public boolean canAccept() {
		boolean result = true;
		if (ghostCard != null) {
			for (WhiteboardTeam team : teams) {
				if (team.contains(ghostCard.getPosition())) {
					result = team.canAccept(ghostCard);
					break;
				}
			}
		}
		return result;
	}
	
	private boolean updateTeamsBasedOnNewCardPosition(WhiteboardCard card) {
		boolean changed = false;
		
		//check if we landed close to any existing companionships
		for (WhiteboardTeam team : teams) {
			if (team.canAccept(card)) {
				//Main.say(card + " is close to " + team);
				team.add(card);
				changed = true;
				break;
			}
		}
		
		return changed;
	}
	
	public void saveState(ObjectOutputStream s) throws Exception {
		s.writeObject(allTeams);
		s.writeObject(chartStyle);
	}
	
	@SuppressWarnings("unchecked")
	public void restoreState(ObjectInputStream s) throws Exception {
		allTeams = (Map<Auxiliary, List<WhiteboardTeam>>)s.readObject();
		chartStyle = (ChartStyle)s.readObject();
	}
		
	private boolean doesCardExist(WhiteboardCard card) {
		for (WhiteboardTeam t : teams) {
			if (t.contains(card)) {
				return true;
			}
		}
		return false;
	}
	
	//when we select a card, both it and its affiliated
	//team get selected. However, if the card no longer
	//exists in the whiteboard, then it's been moved back
	//to the pool. In that case, check the selected team.
	//Is it empty? Then delete it from the list of teams.
	private void select(WhiteboardCard card) {
		if (selectedCard != null) {
			if (!doesCardExist(selectedCard)) {
				if (teamContainingSelectedCard.isEmpty()) {
					teams.remove(teamContainingSelectedCard);
				}
			}
		}
		selectedCard = card;
		teamContainingSelectedCard = selectedCard.getAffiliatedTeam();
		//Main.say("Selected Card=" + selectedCard.getName());
	}
	
	public void refreshCards() {
		List<Team> companionshipsNotAccountedFor = new ArrayList<Team>();
		for (Project p : ward.getAllProjects()) {
			Team comp = p.getTeam();
			if (comp != null) {
				companionshipsNotAccountedFor.add(comp);
			}
		}

		//Check if any cards/teams no longer exist, and delete them.
		List<WhiteboardTeam> condemned = new ArrayList<WhiteboardTeam>();
		for (WhiteboardTeam team : teams) {
			team.compareWithWard(ward, getGraphics2D());
			//check if team is empty; if so, delete it.
			if (team.isEmpty()) condemned.add(team);
			//if the team exists in cards, then its companionship *is* accounted for
			companionshipsNotAccountedFor.remove(team.getTeam());
		}
		for (WhiteboardTeam team : condemned) {
			teams.remove(team);
		}
		
		//Check if any teams are not currently represented in the whiteboard.
		//Add them.
		for (Team comp : companionshipsNotAccountedFor) {
			//first, find the bounding box of all the whiteboardteams.
			//Then build our new team, putting it flush to the top,
			//and to the right of the bounding box. (why not?)
			//FIXME the positioning of the new companionship is
			//not quite right. Fix if you have time.
			Point dummy = new Point();
			Rectangle bbox = WhiteboardTeam.computeUnion(teams);
			WhiteboardTeam newTeam = new WhiteboardTeam(this, dummy);
			newTeam.setCompanionshipManually(comp);
			Project project = comp.getProject();
			if (project != null) {
				newTeam.softAdd(new WhiteboardCard(project, Project.getTeachingStatus(), dummy, getGraphics2D()));
			}
			for (Resource r : comp.getResources()) {
				newTeam.softAdd(new WhiteboardCard(r, Resource.getTeachingStatus(), dummy, getGraphics2D()));
			}
			newTeam.setTopLeftCorner(bbox.x+bbox.width+Config.wb_distanceBetweenCards, bbox.y);
			teams.add(newTeam);
		}
		
		repaint();
	}
	
	public void setCurrentAuxiliary(Auxiliary aux) {
		teams = allTeams.get(aux);
		repaint();
	}

	//TODO this "ghost card" business is just a bandaid hack.
	//A more elegant way would just be to create the "selected
	//card" early, then change it to a non-ghost when it drops.
	public void updateGhostCard(Entity entity, EntityStatus status, Point pos) {
		if (entity == null) {
			if (ghostCard != null) {
				ghostCard = null;
				repaint();
			}
		} else {
			if (ghostCard == null) {
				ghostCard = new WhiteboardCard(entity, status, pos,
						getGraphics2D());
				ghostCard.setGhostStatus(true);
			}
			ghostCard.setPosition(pos);
			repaint();
		}
	}

	@Override
	protected void drawAdditional(Graphics2D g) {
		if (teams.isEmpty()) {
			//If there are no cards currently on the whiteboard,
			//then print a friendly helper message. This serves
			//two purposes. First, it's a nice thing for beginning
			//users. Second, it gives the Graphics object a head
			//start on initializing itself (in Linux, anyway) so
			//that the user doesn't experience weird DnD errors
			//when dragging out the first card.
			g.setFont(Config.largeFont);
			g.setColor(Config.textColor);
			final int FONT_HEIGHT = g.getFontMetrics().getHeight();
			final int LEFT_MARGIN = 20;
			final int SPACE_BETWEEN_WORDS = 6;
			int x = LEFT_MARGIN;
			int y = FONT_HEIGHT;
			int width;
			String token, msg;
			if (ward.getAllProjects().isEmpty() && ward.getAllResources().isEmpty()) {
				msg = Main.messages.getString(Constants.WHITEBOARD_INTRO_MSG1);
			} else {
				msg = Main.messages.getString(Constants.WHITEBOARD_INTRO_MSG2);
			}
			Scanner s = new Scanner(msg);
			
			//make sure the text wraps correctly
			while (s.hasNext()) {
				token = s.next();
				width = g.getFontMetrics().stringWidth(token);
				if ((x+width) > windowSize.width) {
					x = LEFT_MARGIN;
					y += FONT_HEIGHT;
				}
				g.drawString(token, x, y);
				x += (width + SPACE_BETWEEN_WORDS);
			}
			return;
		}
		
		//draw the "ghost card" if applicable.
		if (ghostCard != null) {
			ghostCard.draw(g);
		}
	}

}
